extern crate hakuban;
use std::{sync::Arc, time::Duration};

use futures::{stream::futures_unordered::FuturesUnordered, StreamExt};
use tokio::{sync::{mpsc, oneshot}, task::JoinHandle};

use super::TypeMap;
use crate::scene::*;

pub struct TestDirector;

impl TestDirector {
	pub fn spawn(scene: Arc<Scene>, _name: String, types: TypeMap) -> (mpsc::Sender<Order>, JoinHandle<()>) {
		let (tx, mut rx) = mpsc::channel::<Order>(1000);
		let join_handle = tokio::spawn(async move {
			let mut children = vec![];
			while let Some(order) = rx.recv().await {
				match order {
					Order::Event(frame, event, done) => match event {
						TestAction::Create { actor, processor } => {
							let (order_tx, join_handle) = (types.get(processor.as_str()).unwrap())(scene.clone(), actor.clone());
							children.push((order_tx.clone(), join_handle));
							scene.actors.write().unwrap().insert(actor.to_owned(), order_tx);
							done.send(Ok(())).unwrap();
						}
						TestAction::Run { id, repeat } => {
							let scene_for_run = scene.clone();
							tokio::spawn(async move {
								for iteration in 0..repeat.unwrap_or(1) {
									let block = scene_for_run.scripts.get(&id).unwrap().clone();
									match block {
										Block::Test(test) | Block::Fragment(test) =>
											for (step_index, events_by_actor) in test.steps.iter().enumerate() {
												let dones = FuturesUnordered::new();
												for (actor_index, (actor_name, actor_events)) in events_by_actor.iter().enumerate() {
													if actor_index > 1 {
														print!("\t")
													};

													let actor_tx = scene_for_run
														.actors
														.read()
														.unwrap()
														.get(actor_name)
														.unwrap_or_else(|| panic!("Unknown actor: {}", actor_name))
														.clone();

													let (all_done_tx, all_done_rx) = oneshot::channel();
													dones.push(all_done_rx);
													let base_frame = frame.clone();
													let fragment_id = id.clone();
													let events = actor_events.clone();
													let name = actor_name.clone();
													let index = step_index;
													tokio::spawn(async move {
														for event in events.iter() {
															let new_frame = Arc::new(StackFrame {
																rest: Some(base_frame.clone()),
																top: RunFrame {
																	fragment_id: fragment_id.clone(),
																	step_number: index + 1,
																	repetition: if repeat.is_some() { iteration + 1 } else { 0 },
																},
															});
															println!("• {} / {} / {:?}", new_frame.path(), name, event);
															let (done_tx, done_rx) = oneshot::channel();
															actor_tx.send(Order::Event(new_frame.clone(), event.clone(), done_tx)).await.unwrap();
															done_rx.await.unwrap().unwrap();
															println!("• {} / {} / 🗸", new_frame.path(), name);
														}
														all_done_tx.send(Ok(())).unwrap();
													});
												}

												let errors: Vec<String> = dones
													.collect::<Vec<Result<Result<(), String>, oneshot::error::RecvError>>>()
													.await
													.into_iter()
													.filter_map(|result| result.unwrap().err())
													.collect();
												if !errors.is_empty() {
													for error in errors {
														print!("shit: {}", error);
													}
													panic!("gg");
												};

												tokio::time::sleep(Duration::from_secs_f32(scene_for_run.interstep_delay)).await;
												println!("\n");
											},
									}
								}
								done.send(Ok(())).unwrap();
							});
						}
						_ => panic!(),
					},
					Order::Exit => {
						for (order_tx, join_handle) in children {
							order_tx.send(Order::Exit).await.unwrap();
							join_handle.await.unwrap();
						}
						return;
					}
				}
			}
		});
		(tx, join_handle)
	}
}
