use std::{collections::HashSet, sync::Arc};

use super::core::TagCore;
use crate::{contract::Contract, events::EventMerge, node::core::NodeCore, object::{core::ObjectVersion, observe::{ObjectDeserializer, ObjectObserveState}}, observe_contract::ObserveContract, DefaultSerializerError, LocalNode, ObjectDescriptor, TagDescriptor};

/// Represents a wish, a __contract__ to observe all objects with specific tag
pub struct TagObserve<T: Send + Sync + 'static, E = DefaultSerializerError> {
	contract: Arc<TagObserveContract>,
	deserializer: ObjectDeserializer<T, E>,
}

impl<T: Send + Sync + 'static, E> TagObserve<T, E> {
	pub(super) fn new(contract: Arc<TagObserveContract>, deserializer: ObjectDeserializer<T, E>) -> TagObserve<T, E> {
		TagObserve { contract, deserializer }
	}

	pub fn descriptor(&self) -> &TagDescriptor {
		&self.contract.tag_core.descriptor
	}

	pub fn object_descriptors(&self) -> HashSet<ObjectDescriptor> {
		self.contract.tag_core.active_objects.read().unwrap().keys().cloned().collect()
	}

	pub fn changes(&self) -> EventMerge<ObjectDescriptor> {
		self.contract.tag_core.active_objects_changes()
	}

	pub fn object_state<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>) -> Result<Option<ObjectObserveState<T>>, E> {
		let object_core = self.contract.tag_core.active_objects.read().unwrap().get(descriptor.into()).cloned();
		if let Some(object_core) = object_core {
			let state = object_core.state_get();
			let cooked = if let Some(raw) = state.3.as_ref() { Some((self.deserializer)(state.2.as_ref().unwrap(), raw)?) } else { None };
			Ok(Some(ObjectObserveState { synchronized: state.0, data_version: state.1.clone(), data: cooked, data_type: state.2 }))
		} else {
			Ok(None)
		}
	}

	pub fn object_data<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>) -> Result<Option<T>, E> {
		Ok(self.object_state(descriptor)?.and_then(|state| state.data))
	}

	pub fn object_version<'a>(&self, descriptor: impl Into<&'a ObjectDescriptor>) -> Option<ObjectVersion> {
		let object_core = self.contract.tag_core.active_objects.read().unwrap().get(descriptor.into()).cloned();
		object_core.and_then(|object_core| object_core.version_get())
	}
}

impl<T: Sync + Send, E> Drop for TagObserve<T, E> {
	fn drop(&mut self) {
		self.contract.destroy();
	}
}



pub(super) struct TagObserveContract {
	pub(super) local_node: Option<LocalNode>,
	pub(super) tag_core: Arc<TagCore>,
}

impl TagObserveContract {
	pub(super) fn new(local_node: LocalNode, tag_core: Arc<TagCore>) -> Arc<TagObserveContract> {
		let contract = Arc::new(TagObserveContract { local_node: Some(local_node), tag_core });
		contract.tag_core.link_tag_observe_contract(contract.clone());
		contract
	}

	pub(crate) fn destroy(self: &Arc<Self>) {
		let dyn_self: Arc<dyn ObserveContract> = self.clone();
		self.tag_core.unlink_tag_observe_contract(&dyn_self);
		drop(dyn_self);
		assert_eq!(Arc::strong_count(self), 1);
	}
}


impl Contract for TagObserveContract {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		self.local_node.as_ref().unwrap().shared.node_core.as_ref().unwrap()
	}
}

impl ObserveContract for TagObserveContract {}


impl std::fmt::Debug for TagObserveContract {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("TagObserveContract:{}", self.tag_core.descriptor))
	}
}
