use std::sync::Arc;

use super::core::{ObjectCore, ObjectRawData, ObjectVersion};
use crate::{contract::Contract, events::EventSource, node::core::NodeCore, observe_contract::ObserveContract, DefaultSerializerError, LocalNode, ObjectDescriptor, ObjectType};

pub struct ObjectObserveState<T> {
	pub synchronized: bool,
	pub data_version: Option<ObjectVersion>,
	pub data_type: Option<ObjectType>,
	pub data: Option<T>,
}

/// `<T, E> = Box<dyn Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>`
pub type ObjectDeserializer<T, E> = Box<dyn Send + Sync + Fn(&ObjectType, &ObjectRawData) -> Result<T, E>>;

/// Represents a wish, a __contract__ to observe an object
pub struct ObjectObserve<T, E = DefaultSerializerError> {
	contract: Arc<ObjectObserveContract>,
	deserializer: ObjectDeserializer<T, E>,
}

//TODO: maybe we don't need the T to be all of that any more?
impl<T: Send + Sync + 'static, E> ObjectObserve<T, E> {
	pub(super) fn new(contract: Arc<ObjectObserveContract>, deserializer: ObjectDeserializer<T, E>) -> ObjectObserve<T, E> {
		ObjectObserve { contract, deserializer }
	}

	pub fn descriptor(&self) -> &ObjectDescriptor {
		&self.contract.object_core.descriptor
	}

	pub fn changes(&self) -> EventSource<ObjectDescriptor> {
		self.contract.object_core.changes()
	}

	pub fn synchronized(&self) -> bool {
		self.contract.object_core.synchronized()
	}

	pub fn object_state(&self) -> Result<ObjectObserveState<T>, E> {
		let state = self.contract.object_core.state_get();
		let cooked = if let Some(raw) = state.3.as_ref() { Some((self.deserializer)(state.2.as_ref().unwrap(), raw)?) } else { None };
		Ok(ObjectObserveState { synchronized: state.0, data_version: state.1.clone(), data: cooked, data_type: state.2 })
	}

	pub fn object_data(&self) -> Result<Option<T>, E> {
		Ok(self.object_state()?.data)
	}

	pub fn object_version(&self) -> Option<ObjectVersion> {
		self.contract.object_core.version_get()
	}
}

impl<T, E> Drop for ObjectObserve<T, E> {
	fn drop(&mut self) {
		self.contract.destroy();
	}
}



pub(super) struct ObjectObserveContract {
	pub local_node: Option<LocalNode>,
	pub object_core: Arc<ObjectCore>,
}

impl ObjectObserveContract {
	pub(super) fn new(local_node: LocalNode, object_core: Arc<ObjectCore>) -> Arc<ObjectObserveContract> {
		let contract = Arc::new(ObjectObserveContract { local_node: Some(local_node), object_core });
		contract.object_core.link_object_observe_contract(contract.clone());
		contract
	}

	pub(super) fn destroy(self: &Arc<Self>) {
		let dyn_self: Arc<dyn ObserveContract> = self.clone();
		self.object_core.unlink_object_observe_contract(&dyn_self);
		drop(dyn_self);
		assert_eq!(Arc::strong_count(self), 1);
	}
}



impl Contract for ObjectObserveContract {
	fn id(&self) -> usize {
		(self as *const Self) as usize
	}

	fn node_core(&self) -> &Arc<NodeCore> {
		self.local_node.as_ref().unwrap().shared.node_core.as_ref().unwrap()
	}
}

impl ObserveContract for ObjectObserveContract {}


impl std::fmt::Debug for ObjectObserveContract {
	fn fmt(&self, formatter: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		formatter.write_str(&format!("ObjectObserveContract:{}", self.object_core.descriptor))
	}
}
