use std::{hash::{Hash, Hasher}, sync::{atomic::AtomicU32, Arc, RwLock}};


//TODO: remove pubs inside, implement accessors, then make NodeCore only hold Pin<Arc<NodeCoreInternal>>

pub(crate) struct NodeCore {
	pub upstream: bool,
	pub capacity: AtomicU32,
	pub limit: AtomicU32,
	pub load: AtomicU32,
	pub name: RwLock<Option<String>>,
}


impl NodeCore {
	pub fn new(upstream: bool, capacity: Option<u32>, limit: Option<u32>, name: Option<String>) -> Arc<NodeCore> {
		let capacity = capacity.or_else(|| Some(num_cpus::get() as u32 * 1000)).unwrap();
		Arc::new(NodeCore {
			upstream,
			capacity: AtomicU32::new(capacity),
			limit: AtomicU32::new(limit.or(Some(capacity * 16)).unwrap()),
			load: AtomicU32::new(0),
			name: RwLock::new(name),
		})
	}

	pub fn name_set(&self, name: &str) {
		*self.name.write().unwrap() = Some(name.to_string());
	}

	//TODO: this is nasty. it should work as long as NodeCore is always in Arc, and nobody messes up with that Arc too much.
	//And passing Pin<Arc<NodeCore>> everywhere is pita. There is probably better way to ensure stability of this address.
	//Maybe hide Pin+Arc inside a wrapper object which would be passed around instead of Arc<NodeCore> ?
	pub fn id(&self) -> usize {
		(self as *const Self) as usize
	}
}


impl std::fmt::Debug for NodeCore {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		if let Some(ref name) = *self.name.read().unwrap() {
			f.write_fmt(format_args!("Node#{:8X}({})", self.id(), name))
		} else {
			f.write_fmt(format_args!("Node#{:8X}()", self.id()))
		}
	}
}

impl Hash for NodeCore {
	fn hash<H: Hasher>(&self, state: &mut H) {
		self.id().hash(state);
	}
}

impl PartialEq for NodeCore {
	fn eq(&self, other: &Self) -> bool {
		self.id() == other.id()
	}
}

impl Eq for NodeCore {}
