/*! Communication primitives; only for tests and custom transports
*/

use serde::{Deserialize, Serialize};

use crate::{descriptor::ObjectDescriptor, object::core::{ObjectRawData, ObjectVersion}, ObjectType, TagDescriptor};

#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
pub struct Message {
	pub sequence_number: u64,
	pub acknowledgment_number: u64,
	pub changes: Vec<Change>,
}

// this was just plain enum before, with struct-like fields.
// but serde doesn't seem to handle well missing fields in such enum values
// and it's less painful to create and mutate nested structs than such pseudo-structs
#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
#[serde(tag = "change")]
pub enum Change {
	Options(OptionsChange),
	Object(ObjectChange),
	Tag(TagChange),
}

#[derive(Debug, Serialize, Deserialize, PartialEq, Clone)]
pub struct OptionsChange {
	pub name: Option<String>,
	pub custom: Option<serde_json::Value>,
	pub expose_capacity: Option<u32>,
	pub expose_load_limit: Option<u32>,
	pub diff_request: Option<Option<bool>>,
}


#[derive(Debug, Serialize, Deserialize, Eq, PartialEq, Clone)]
pub enum DataChange {
	Data { data: ObjectRawData },
	Diff { old_data_version: Option<ObjectVersion>, diff: ObjectRawData },
}

#[derive(Debug, Serialize, Deserialize, Eq, PartialEq, Clone)]
pub struct ObjectChange {
	pub descriptor: ObjectDescriptor,
	pub observer: Option<bool>,
	pub exposer: Option<bool>,
	pub data: Option<DataChange>,
	pub data_type: Option<ObjectType>,
	pub data_version: Option<ObjectVersion>,
	pub cost: Option<Option<u32>>,
}



#[derive(Debug, Serialize, Deserialize, Eq, PartialEq, Clone)]
pub struct TagChange {
	pub descriptor: TagDescriptor,
	pub observer: Option<bool>,
	pub exposer: Option<bool>,
	pub cost: Option<Option<u32>>,
}

impl ObjectChange {
	pub fn is_empty(&self) -> bool {
		!(self.observer.is_some() || self.exposer.is_some() || self.data.is_some() || self.data_type.is_some() || self.data_version.is_some())
	}
}

impl TagChange {
	pub fn is_empty(&self) -> bool {
		!(self.observer.is_some() || self.exposer.is_some())
	}
}

impl Message {
	pub fn inspect(&self) -> String {
		let mut ret = format!("├───── Message seq:{} ack:{} changes:\n", self.sequence_number, self.acknowledgment_number);
		for change in &self.changes {
			ret.push_str(change.inspect().as_str());
		}
		ret
	}
}

impl Change {
	pub fn inspect(&self) -> String {
		match &self {
			Change::Options(change) => format!(
				"│ Options -> name: {:?}  custom: {:?}  expose_capacity: {:?}  expose_load_limit: {:?}  diff_request: {:?}\n",
				change.name, change.custom, change.expose_capacity, change.expose_load_limit, change.diff_request
			),
			Change::Object(change) => format!(
				"│ Object -> {:?}  observer: {:?}  exposer: {:?}  data_type: {:?}  data_version: {:?}  cost: {:?}\n",
				change.descriptor, change.observer, change.exposer, change.data_type, change.data_version, change.cost
			),
			Change::Tag(change) =>
				format!("│ Tag -> {:?}  observer: {:?}  exposer: {:?}  cost: {:?}\n", change.descriptor, change.observer, change.exposer, change.cost),
		}
	}
}

#[cfg(test)]
mod tests {

	use std::sync::Arc;

	use super::*;

	#[test]
	fn can_serialie_and_deserialize_message() {
		let message = Message {
			sequence_number: 1,
			acknowledgment_number: 2,
			changes: vec![Change::Object(ObjectChange {
				descriptor: (vec!["{}"], "{}").into(),
				observer: Some(true),
				exposer: Some(true),
				data: Some(DataChange::Data { data: Arc::new(b"aaaaa".to_vec()) }),
				data_type: None,
				data_version: None,
				cost: None,
			})],
		};

		//JSON
		let serialized = serde_json::to_string(&message).unwrap();
		println!("{}", serialized);
		let deserialized: Message = serde_json::from_str(&serialized).unwrap();
		assert_eq!(message, deserialized);

		//YAML
		let serialized = serde_yaml::to_string(&message).unwrap();
		println!("{}", serialized);
		let deserialized: Message = serde_yaml::from_str(&serialized).unwrap();
		assert_eq!(message, deserialized);

		//MessagePack
		let serialized = rmp_serde::to_vec(&message).unwrap();
		//dbg!(&serialized);
		let deserialized = rmp_serde::from_read_ref::<Vec<u8>, Message>(&serialized).unwrap();
		assert_eq!(message, deserialized);
	}
}
