use std::{convert::TryFrom, ffi::CString, mem::forget};

use crate::{ffi::FFIError, ObjectDescriptor, TagDescriptor};

#[derive(Clone)]
#[repr(C)]
pub struct FFIObjectDescriptor {
	pub tags_count: usize,
	pub tags: *mut *mut i8,
	pub json: *mut i8,
}


pub(super) fn c_string_to_json_value(json_c: *const i8) -> Result<serde_json::Value, FFIError> {
	let json_str: &str = match unsafe { std::ffi::CStr::from_ptr(json_c).to_str() } {
		Ok(string) => string,
		Err(_error) => {
			return Err(FFIError::InvalidString);
		}
	};
	let json: serde_json::Value = match serde_json::from_str(json_str) {
		Ok(descriptor) => descriptor,
		Err(_error) => {
			return Err(FFIError::InvalidJSON);
		}
	};
	Ok(json)
}

impl TryFrom<&FFIObjectDescriptor> for ObjectDescriptor {
	type Error = FFIError;

	fn try_from(ffi_descriptor: &FFIObjectDescriptor) -> Result<Self, Self::Error> {
		let tags_c_slice: &[*const i8] = unsafe { std::slice::from_raw_parts(ffi_descriptor.tags as *const *const i8, ffi_descriptor.tags_count) };
		let tags_json = tags_c_slice.iter().map(|tag_c| c_string_to_json_value(*tag_c)).collect::<Result<Vec<serde_json::Value>, FFIError>>()?;
		let json = c_string_to_json_value(ffi_descriptor.json)?;
		Ok(ObjectDescriptor::new(tags_json, json))
	}
}

impl From<&ObjectDescriptor> for FFIObjectDescriptor {
	fn from(object_descriptor: &ObjectDescriptor) -> Self {
		let mut tag_strings: Vec<*mut i8> = object_descriptor.tags.iter().map(|tag| CString::new(tag.to_string()).unwrap().into_raw() as *mut i8).collect();
		tag_strings.shrink_to_fit();
		let ret = FFIObjectDescriptor {
			tags_count: tag_strings.len(),
			tags: tag_strings.as_mut_ptr(),
			json: CString::new(object_descriptor.json.to_string()).unwrap().into_raw(),
		};
		forget(tag_strings);
		ret
	}
}

impl FFIObjectDescriptor {
	pub(super) fn free(self) {
		drop(unsafe { CString::from_raw(self.json) });
		let mut tag_strings: Vec<*mut i8> = unsafe { Vec::from_raw_parts(self.tags, self.tags_count, self.tags_count) };
		for string in tag_strings.drain(..) {
			drop(unsafe { CString::from_raw(string) });
		}
		drop(tag_strings);
	}
}


#[repr(C)]
pub struct FFITagDescriptor {
	pub json: *mut i8,
}


impl TryFrom<&FFITagDescriptor> for TagDescriptor {
	type Error = FFIError;

	fn try_from(ffi_descriptor: &FFITagDescriptor) -> Result<Self, Self::Error> {
		let json = c_string_to_json_value(ffi_descriptor.json)?;
		Ok(TagDescriptor::new(json))
	}
}

impl From<&TagDescriptor> for FFITagDescriptor {
	fn from(tag_descriptor: &TagDescriptor) -> Self {
		FFITagDescriptor { json: CString::new(tag_descriptor.json.to_string()).unwrap().into_raw() }
	}
}

// This one's never used right now. There are no ffi functions which send FFITagDescriptors to the other side.
//impl FFITagDescriptor {
//	pub(crate) fn free(self) {
//		drop(unsafe { CString::from_raw(self.json) });
//	}
//}
