use clap::{App, Arg};
use hakuban::{tokio::{WebsocketConnector, WebsocketListener}, LocalNode};
use log::info;
use url::Url;


fn main() -> Result<(), Box<dyn std::error::Error>> {
	env_logger::Builder::from_env(env_logger::Env::default().default_filter_or("hakuban=info")).init();

	let arguments = App::new("Hakuban router")
		.version(env!("CARGO_PKG_VERSION"))
		.about("WebSocket server, routing hakuban objects and tags between clients.\n")
		.arg(
			Arg::with_name("bind")
				.short("b")
				.long("bind")
				.value_name("address")
				.help("address to bind to")
				.default_value("ws://127.0.0.1:3001")
				.takes_value(true),
		)
		.arg(Arg::with_name("connect").short("c").long("connect").value_name("address").help("address to connect to").takes_value(true))
		.arg(
			Arg::with_name("threads")
				.short("t")
				.long("threads")
				.value_name("threads")
				.help("number of cpu threads to use. defaults to number of cores available to the system.")
				.takes_value(true),
		)
		.arg(Arg::with_name("name").short("n").long("node-name").value_name("name").help("node name").default_value("router").takes_value(true))
		.get_matches();

	let bind_address = Url::parse(arguments.value_of("bind").unwrap())?;
	let connect_address = if let Some(connect_address_str) = arguments.value_of("connect") { Some(Url::parse(connect_address_str)?) } else { None };

	let mut runtime_builder = tokio::runtime::Builder::new_multi_thread();
	runtime_builder.enable_all();
	if let Some(threads_string) = arguments.value_of("threads") {
		let threads_int = threads_string.parse::<usize>().map_err(|error| format!("Can't parse thread count value. {:?}", error))?;
		if threads_int == 0 {
			return Err("Threads param can't be 0".into());
		}
		runtime_builder.worker_threads(threads_int);
	}
	let runtime = runtime_builder.build()?;


	runtime
		.block_on(async {
			let local_node = LocalNode::builder().with_name(arguments.value_of("name").unwrap()).build();

			let connector =
				if let Some(connect_address) = connect_address { Some(WebsocketConnector::new(local_node.clone(), connect_address)?) } else { None };

			let listener = WebsocketListener::new(local_node.clone(), bind_address).await?;

			use std::io::BufRead;
			for _line in std::io::stdin().lock().lines() {
				eprintln!();
				eprintln!("{}", local_node.inspect());
				eprintln!("{}", listener.inspect());
				if let Some(connector) = &connector {
					eprintln!("{}", connector.inspect());
				}
			}

			info!("stdin EOF, exiting");
			Ok(()) as Result<(), Box<dyn std::error::Error>>
		})
		.unwrap();

	Ok(())
}
