use std::fmt::Write;

use color_eyre::{eyre::WrapErr, Result, Section};
use figment::{
    providers::{Format, Json, Toml},
    Figment,
};
use lettre::{
    message::{Message, SinglePart},
    transport::{smtp::authentication::Credentials, smtp::SmtpTransport, Transport},
};
use serde::{Deserialize, Serialize};
use tinytemplate::error::Error as TtError;
use ureq::SerdeValue;

#[derive(Deserialize)]
struct Config {
    user_id: String,
    api_key: String,
    email: String,
    email_password: String,
}

const CREATOR_USER_ID: &str = "7fff7603-6003-4008-a650-3c4fcfc1ad18";

fn main() -> Result<()> {
    color_eyre::install()?;

    let config: Config = Figment::new()
        .merge(Toml::file("config.toml"))
        .merge(Json::file("config.json"))
        .extract()
        .wrap_err("Failed to read config")
        .suggestion(
            "Configuration format is described at
            `https://github.com/esposm03/habitica_weekly_report#usage`",
        )?;

    let mut tasks: Response<Vec<Task>> = ureq::get("https://habitica.com/api/v3/tasks/user")
        .set("X-Api-Key", &config.api_key)
        .set("X-Api-User", &config.user_id)
        .set("X-Client", &format!("{}-HabiticaBot", CREATOR_USER_ID))
        .call()
        .wrap_err("Failed to fetch data from Habitica")?
        .into_json()
        .wrap_err("Failed to fetch data from Habitica")?;
    tasks.data.sort_by_key(|t| t.value.round() as i32);

    let rendered = {
        let mut tt = tinytemplate::TinyTemplate::new();
        let context = TemplateContext {
            worst_tasks: tasks.data.iter().take(3).cloned().collect(),
            best_tasks: tasks.data.iter().rev().take(3).cloned().collect(),
        };

        tt.set_default_formatter(&number_to_int);
        tt.add_template("template", include_str!("template.html"))
            .wrap_err("Failed to render template")?;
        tt.render("template", &context)
            .wrap_err("Failed to render template")?
    };

    let email = config
        .email
        .parse::<lettre::message::Mailbox>()
        .wrap_err("Failed to parse email address")?;
    let email = Message::builder()
        .from(email.clone())
        .to(email)
        .subject("Habitica report")
        .singlepart(SinglePart::html(rendered))
        .wrap_err("Failed to send email")?;

    SmtpTransport::relay("smtp.gmail.com")
        .wrap_err("Failed to send email")?
        .credentials(Credentials::new(config.email, config.email_password))
        .build()
        .send(&email)
        .wrap_err("Failed to send email")?;

    Ok(())
}

fn number_to_int(v: &SerdeValue, output: &mut String) -> Result<(), TtError> {
    if let SerdeValue::Number(n) = v {
        let n = n.as_f64().ok_or(TtError::GenericError {
            msg: "A number could not be formatted as a string".into(),
        })?;
        write!(output, "{:.2}", n)?;
        Ok(())
    } else {
        tinytemplate::format(v, output)
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
struct Response<T> {
    success: bool,
    app_version: String,
    data: T,
}

#[derive(Serialize)]
struct TemplateContext {
    worst_tasks: Vec<Task>,
    best_tasks: Vec<Task>,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
struct Task {
    r#type: TaskType,
    value: f64,
    text: String,
}

#[derive(Deserialize, Serialize, Debug, Clone, Copy)]
#[serde(rename_all = "camelCase")]
enum TaskType {
    Todo,
    Habit,
    Daily,
}
