use reqwest;
use reqwest::header::HeaderMap;
use reqwest::Client;
use serde_json::Value;
use std::fmt;
use crate::task::api_credentials::ApiCredentials;
use serde_json::json;
use async_trait::async_trait;

#[async_trait]
pub trait RestOperations {
    async fn get(&self, url: &str) -> Result<Value, RestClientError>;
}

pub struct RestClient {
    api_credentials: ApiCredentials,
    client: Client,
}

impl RestClient {
    pub fn new(api_credentials: ApiCredentials) -> Box<dyn RestOperations> {
        let client = reqwest::Client::new();
        Box::new(RestClient {
            api_credentials,
            client,
        })
    }

    fn build_auth_headers(&self) -> HeaderMap {
        let api_credentials = self.get_api_credentials();
        let mut headers = HeaderMap::new();

        headers.insert("x-api-user", api_credentials.get_user().to_owned().parse().unwrap());
        headers.insert("x-api-key", api_credentials.get_key().to_owned().parse().unwrap());

        headers
    }

    fn get_api_credentials(&self) -> &ApiCredentials {
        &self.api_credentials
    }

    fn get_client(&self) -> &Client {
        &self.client
    }
}

#[async_trait]
impl RestOperations for RestClient {
    async fn get(&self, url: &str) -> Result<Value, RestClientError> {
        let headers = self.build_auth_headers();

        let response = self
            .get_client()
            .get(url)
            .headers(headers)
            .send()
            .await
            .unwrap();

        Ok(json!(response.text().await.unwrap()))
    }
}

pub struct RestClientError {
    message: String,
}

impl fmt::Debug for RestClientError {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(formatter, "Error")
    }
}

impl RestClientError {
    pub fn new(message: String) -> RestClientError {
        RestClientError { message }
    }

    pub fn get_message(&self) -> &String {
        &self.message
    }
}
