use either::Either;
use ggez::Context;
use h1v3_logic::{Action, ActionKind, Cell, Color as HiveColor};
use log::*;

use crate::{consts::*, MainState};

impl MainState {
    pub fn check_for_new_piece_placement(&mut self, ctx: &Context, x: f32, y: f32) -> bool {
        let (player_color, piece_idx) =
            if let Some(Either::Left(selected_new_piece)) = self.selected_piece {
                selected_new_piece
            } else {
                return false;
            };

        let pos = self.point_to_position(ctx, x, y);
        if pos.is_none() {
            return false;
        }
        let pos = pos.unwrap();

        let piece = self.game.players[player_color as usize].pieces[piece_idx].clone();

        trace!(
            "attempting to place a new piece ({:?}) at position {}",
            piece,
            pos
        );

        let action = Action {
            player_color,
            kind: ActionKind::Placement(piece, pos),
        };

        match self.game.progress(action) {
            Ok(status) => {
                trace!("placement succeeded");
                self.curr_moveset = None;
                self.status = status;
            }
            Err(e) => {
                error!("placement failed: {:?}", e);
                return false;
            }
        }
        self.game.players[player_color as usize]
            .pieces
            .remove(piece_idx);
        self.selected_piece = None;

        true
    }

    pub fn check_for_new_piece_selection(&mut self, ctx: &Context, x: f32, y: f32) -> bool {
        let window_height = self.window_dimensions(ctx).1;

        let player_ui_y_range_white = (R, 3.0 * R);
        let player_ui_y_range_black = (window_height - 3.0 * R, window_height - R);

        let player_color = if y > player_ui_y_range_white.0 && y < player_ui_y_range_white.1 {
            HiveColor::White
        } else if y > player_ui_y_range_black.0 && y < player_ui_y_range_black.1 {
            HiveColor::Black
        } else {
            return false;
        };

        if let Some(current_player) = self.game.current_player() {
            if current_player != player_color {
                error!("it's not {:?}'s turn", player_color);
                return false;
            }
        }

        let player_idx = player_color as usize;

        let player_ui_x_range = (
            R * 0.5,
            self.game.players[player_idx].pieces.len() as f32 * R * 2.0 + R * 0.5,
        );
        if !(x > player_ui_x_range.0 && x < player_ui_x_range.1) {
            return false;
        }

        let piece_idx = (0..self.game.players[player_idx].pieces.len())
            .find(|i| x < R + (2.0 * R) * (*i as f32 + 1.0))
            .unwrap();

        if self.selected_piece != Some(Either::Left((player_color, piece_idx))) {
            trace!("selected a new piece: {:?}[{}]", player_color, piece_idx);
            self.selected_piece = Some(Either::Left((player_color, piece_idx)));
        } else {
            trace!("deselected a new piece: {:?}[{}]", player_color, piece_idx);
            self.selected_piece = None;
        }
        self.curr_moveset = None;

        true
    }

    pub fn check_for_placed_piece_move(&mut self, ctx: &Context, x: f32, y: f32) -> bool {
        let selected_piece_pos = if let Some(Either::Right(pos)) = self.selected_piece {
            pos
        } else {
            return false;
        };

        let pos = self.point_to_position(ctx, x, y);
        if pos.is_none() {
            return false;
        }
        let pos = pos.unwrap();

        if selected_piece_pos == pos {
            return false;
        }

        if let Some(player_color) = self
            .game
            .hive
            .cells
            .iter()
            .find(|cell| cell.position == selected_piece_pos)
            .map(Cell::color)
        {
            let action = Action {
                player_color,
                kind: ActionKind::Movement(selected_piece_pos, pos),
            };

            trace!(
                "attempting to move a piece from position {} to {}",
                selected_piece_pos,
                pos
            );

            match self.game.progress(action) {
                Ok(status) => {
                    trace!("move succeeded");
                    self.curr_moveset = None;
                    self.status = status;
                }
                Err(e) => {
                    error!("failed to move: {:?}", e);
                    return false;
                }
            }
            self.selected_piece = None;

            true
        } else {
            false
        }
    }

    pub fn check_for_placed_piece_selection(&mut self, ctx: &Context, x: f32, y: f32) -> bool {
        let pos = self.point_to_position(ctx, x, y);
        if pos.is_none()
            || !self
                .game
                .hive
                .cells
                .iter()
                .any(|cell| Some(cell.position) == pos)
        {
            return false;
        }
        let pos = pos.unwrap();

        if let Some(current_player) = self.game.current_player() {
            if Some(current_player) != self.game.hive.get_cell(pos).map(Cell::color) {
                error!("it's not {:?}'s turn", !current_player);
                return false;
            }
        }

        if self.selected_piece != Some(Either::Right(pos)) {
            trace!("selected a placed piece at position {}", pos);
            self.selected_piece = Some(Either::Right(pos));
        } else {
            trace!("deselected a placed piece at position {}", pos);
            self.selected_piece = None;
        }
        self.curr_moveset = None;

        true
    }

    pub fn check_for_pass(&mut self, ctx: &Context, x: f32, y: f32) -> bool {
        let (window_width, window_height) = self.window_dimensions(ctx);

        let player_ui_y_range_white = (R, 3.0 * R);
        let player_ui_y_range_black = (window_height - 3.0 * R, window_height - R);

        let player_ui_x_range = (window_width - 3.0 * R, window_width);
        if !(x > player_ui_x_range.0 && x < player_ui_x_range.1) {
            return false;
        }

        let player_color = if y > player_ui_y_range_white.0 && y < player_ui_y_range_white.1 {
            HiveColor::White
        } else if y > player_ui_y_range_black.0 && y < player_ui_y_range_black.1 {
            HiveColor::Black
        } else {
            return false;
        };

        if let Some(current_player) = self.game.current_player() {
            if current_player != player_color {
                return false;
            }
        }

        if !self.game.hive.must_pass(player_color) {
            error!("can't pass the turn: {:?} can move", player_color);
            return false;
        }

        let action = Action {
            player_color,
            kind: ActionKind::Pass,
        };

        self.game.progress(action).unwrap();
        trace!("{:?} passes", player_color);

        self.selected_piece = None;
        self.curr_moveset = None;

        true
    }
}
