use std::iter;

use ggez::Context;
use h1v3_logic::{edge_index_non_tip, is_at_ring_tip, ring, ring_offset, tip_index, Position};

use super::{MainState, A, BORDER_WIDTH, R};

pub fn position_to_point(pos: Position, r: f32, center: (f32, f32)) -> (f32, f32) {
    if pos == 0 {
        return center;
    }

    let ring = ring(pos);

    if is_at_ring_tip(pos) {
        let tip_idx = tip_index(pos);
        let ring = ring as f32;

        let (xm, ym) = match tip_idx {
            0 => (0.0, -2.0 * ring),
            1 => (3.0 * ring, -ring),
            2 => (3.0 * ring, ring),
            3 => (0.0, 2.0 * ring),
            4 => (-3.0 * ring, ring),
            5 => (-3.0 * ring, -ring),
            _ => unreachable!(),
        };
        let x = (xm * (r * A.cos())) as f32;
        let y = (ym * (r * A.sin())) as f32;

        (center.0 + x, center.1 + y)
    } else {
        let edge_idx = edge_index_non_tip(pos);
        let ring_offset = ring_offset(ring);
        let ring_pos = pos - ring_offset;
        let tip_offset = ring_pos - edge_idx * ring;
        let tip_pos = ring_offset + edge_idx * ring;
        let ring = tip_offset as f32;

        let tip_point = position_to_point(tip_pos, r, center);

        let (xm, ym) = match (edge_idx + 2) % 6 {
            0 => (0.0, -2.0 * ring),
            1 => (3.0 * ring, -ring),
            2 => (3.0 * ring, ring),
            3 => (0.0, 2.0 * ring),
            4 => (-3.0 * ring, ring),
            5 => (-3.0 * ring, -ring),
            _ => unreachable!(),
        };
        let x = (xm * (r * A.cos())) as f32;
        let y = (ym * (r * A.sin())) as f32;

        (tip_point.0 + x, tip_point.1 + y)
    }
}

impl MainState {
    pub fn ring_range(&self) -> (usize, usize) {
        let mut min_ring = 100;
        let mut max_ring = 0;

        for pos in self.game.hive.cells.iter().map(|cell| cell.position) {
            let ring = ring(pos);
            if ring < min_ring {
                min_ring = ring;
            } else if ring > max_ring {
                max_ring = ring;
            }
        }

        (min_ring, max_ring)
    }

    pub fn get_hex_radius(&self, ctx: &Context) -> f32 {
        let (ww, wh) = self.window_dimensions(ctx);

        let (_min_ring, max_ring) = self.ring_range();

        R.max(wh.min(ww) / ((max_ring + 2) * 2) as f32 * 0.5)
    }

    pub fn point_to_position(&self, ctx: &Context, x: f32, y: f32) -> Option<Position> {
        let r = self.get_hex_radius(ctx);
        let r_small = r * A.sin() - BORDER_WIDTH;
        let center = self.window_center(ctx);

        for pos in self.game.hive.cells.iter().map(|cell| cell.position).chain(
            self.game
                .hive
                .cells
                .iter()
                .flat_map(|cell| {
                    self.game
                        .hive
                        .cell_neighboring_positions(cell.position)
                        .into_iter()
                })
                .chain(iter::once(0)),
        ) {
            let cell_center = position_to_point(pos, r, center);

            if (x < cell_center.0 + r_small && x > cell_center.0 - r_small)
                && (y < cell_center.1 + r_small && y > cell_center.1 - r_small)
            {
                return Some(pos);
            }
        }

        None
    }
}
