use std::{collections::HashMap, env, path};

use either::Either;
use ggez::conf::{ModuleConf, WindowMode};
use ggez::event;
use ggez::graphics;
use ggez::{Context, GameResult};
use h1v3_logic::{Color as HiveColor, Game, GameStatus, Position, Species};
use log::*;

mod consts;
use consts::*;

mod input;

mod math;
use math::*;

mod display;

struct MainState {
    game: Game,
    selected_piece: Option<Either<(HiveColor, usize), Position>>, // (new, placed)
    curr_moveset: Option<Vec<Position>>,
    status: GameStatus,
    sprites: HashMap<Species, graphics::Image>,
}

impl Default for MainState {
    fn default() -> MainState {
        MainState {
            game: Game::default(),
            selected_piece: None,
            curr_moveset: None,
            status: GameStatus::Ongoing,
            sprites: Default::default(),
        }
    }
}

impl event::EventHandler<ggez::GameError> for MainState {
    fn mouse_button_down_event(
        &mut self,
        ctx: &mut Context,
        _button: event::MouseButton,
        x: f32,
        y: f32,
    ) {
        // trace!("mouse pressed at ({}, {})", x, y);

        let _ = self.check_for_placed_piece_move(ctx, x, y)
            || self.check_for_pass(ctx, x, y)
            || self.check_for_new_piece_selection(ctx, x, y)
            || self.check_for_placed_piece_selection(ctx, x, y)
            || self.check_for_new_piece_placement(ctx, x, y);
    }

    fn update(&mut self, ctx: &mut Context) -> GameResult {
        match self.status {
            GameStatus::GameOver(color) => {
                info!("{:?} won!", color);
                event::quit(ctx);
            }
            GameStatus::Draw => {
                info!("it's a draw!");
                event::quit(ctx);
            }
            _ => {}
        }

        Ok(())
    }

    fn draw(&mut self, ctx: &mut Context) -> GameResult {
        let window_size = self.window_dimensions(ctx);
        graphics::set_screen_coordinates(
            ctx,
            graphics::Rect::new(0.0, 0.0, window_size.0, window_size.1),
        )?;

        graphics::clear(ctx, BEIGE.into());

        let (player_ui_meshes, ui_piece_positions) = self.create_player_ui_meshes(ctx)?;
        for mesh in player_ui_meshes {
            graphics::draw(ctx, &mesh, graphics::DrawParam::default())?;
        }

        let (hive_meshes, placed_piece_positions) = self.create_hive_mesh(ctx)?;
        for mesh in hive_meshes {
            graphics::draw(ctx, &mesh, graphics::DrawParam::default())?;
        }

        let moveset_meshes = self.create_moveset_mesh(ctx, self.game.current_player())?;
        for mesh in moveset_meshes {
            graphics::draw(ctx, &mesh, graphics::DrawParam::default())?;
        }

        if let Some(mesh) = self.create_selection_mesh(ctx)? {
            graphics::draw(ctx, &mesh, graphics::DrawParam::default())?;
        }

        for (piece, (x, y)) in ui_piece_positions
            .into_iter()
            .chain(placed_piece_positions.into_iter())
        {
            let sprite = &self.sprites[&piece.species];

            let mut param = graphics::DrawParam::default();
            if let graphics::Transform::Values { dest, scale, .. } = &mut param.trans {
                let (factor, offset) = if y > R * 3.0 && y < window_size.1 - R * 3.0 {
                    let r = self.get_hex_radius(ctx);
                    let factor = r / 512.0;
                    let offset = (r / 2.0, r / 2.0);
                    (factor, (offset.0, offset.1))
                } else {
                    (0.07, (R / 1.8, R / 1.8))
                };
                *dest = [x - offset.0, y - offset.1].into();
                *scale = [factor, factor].into();
            }
            graphics::draw(ctx, sprite, param)?;
        }

        graphics::draw_queued_text(
            ctx,
            graphics::DrawParam::default(),
            None,
            graphics::FilterMode::Linear,
        )?;

        graphics::present(ctx)?;
        Ok(())
    }
}

pub fn main() -> GameResult {
    let mut logger = env_logger::builder();
    logger
        .filter(Some("gfx_device_gl"), log::LevelFilter::Warn)
        .filter(Some("ggez"), log::LevelFilter::Warn);
    logger.init();

    let mut cb = ggez::ContextBuilder::new("Hive", "ljedrz").modules(ModuleConf {
        gamepad: false,
        audio: false,
    });

    if let Ok(manifest_dir) = env::var("CARGO_MANIFEST_DIR") {
        let mut path = path::PathBuf::from(manifest_dir);
        path.push("assets");
        debug!("adding resource path {:?}", path);
        cb = cb.add_resource_path(path);
    }

    let (mut ctx, event_loop) = cb.build()?;

    let window_mode = WindowMode {
        width: 800.0,
        height: 800.0,
        ..Default::default()
    };
    graphics::set_mode(&mut ctx, window_mode)?;

    let mut state = MainState::default();
    load_assets(&mut state, &mut ctx)?;

    event::run(ctx, event_loop, state)
}

fn load_assets(state: &mut MainState, ctx: &mut Context) -> GameResult {
    state
        .sprites
        .insert(Species::Beetle, graphics::Image::new(ctx, "/beetle.png")?);
    state.sprites.insert(
        Species::Grasshopper,
        graphics::Image::new(ctx, "/grasshopper.png")?,
    );
    state.sprites.insert(
        Species::QueenBee,
        graphics::Image::new(ctx, "/queen_bee.png")?,
    );
    state.sprites.insert(
        Species::SoldierAnt,
        graphics::Image::new(ctx, "/soldier_ant.png")?,
    );
    state
        .sprites
        .insert(Species::Spider, graphics::Image::new(ctx, "/spider.png")?);
    // state.sprites.insert(Species::Mosquito, graphics::Image::new(ctx, "/mosquito.png")?);
    // state.sprites.insert(Species::Pillbug, graphics::Image::new(ctx, "/pillbug.png")?);
    // state.sprites.insert(Species::Ladybug, graphics::Image::new(ctx, "/ladybug.png")?);

    Ok(())
}
