use super::{consts::*, position_to_point, MainState, A, BEIGE, BORDER_WIDTH, R};
use either::Either;
use ggez::{graphics, Context, GameResult};
use glam::*;
use h1v3_logic::{Cell, Color as HiveColor, Piece};
use log::*;

impl MainState {
    pub fn window_center(&self, ctx: &Context) -> (f32, f32) {
        let (width, height) = self.window_dimensions(ctx);
        (width / 2.0, height / 2.0)
    }

    pub fn window_dimensions(&self, ctx: &Context) -> (f32, f32) {
        graphics::size(ctx)
    }

    pub fn create_hex_mesh(
        &self,
        ctx: &mut Context,
        center: (f32, f32),
        r: f32,
        surface: Either<&Piece, &str>,
        selected: bool,
    ) -> GameResult<graphics::Mesh> {
        let (x, y) = center;
        let tips: Vec<_> = (0..6)
            .map(|i| Vec2::new(x + r * (A * i as f32).cos(), y + r * (A * i as f32).sin()))
            .collect();

        let color = if let Either::Left(piece) = surface {
            if piece.color == HiveColor::White {
                graphics::Color::WHITE
            } else {
                DARK_GREY.into()
            }
        } else {
            BEIGE.into()
        };

        let border_color = if selected {
            graphics::Color::RED
        } else {
            graphics::Color::BLACK
        };

        let mb = &mut graphics::MeshBuilder::new();

        let mut stroke_opts: graphics::StrokeOptions = Default::default();
        stroke_opts.line_width = BORDER_WIDTH;

        let fill_opts: graphics::FillOptions = Default::default();

        mb.polygon(graphics::DrawMode::Fill(fill_opts), &tips, color)?;
        mb.polygon(graphics::DrawMode::Stroke(stroke_opts), &tips, border_color)?;

        /*
        mb.circle(
            graphics::DrawMode::Stroke(Default::default()),
            [center.0, center.1],
            r * A.sin() - BORDER_WIDTH,
            0.1,
            graphics::Color::GREEN,
        )?;
        */

        if let Either::Right(label) = surface {
            let mut text = graphics::Text::new(label);
            text.set_font(Default::default(), graphics::PxScale { x: 20.0, y: 20.0 });
            graphics::queue_text(
                ctx,
                &text,
                [center.0 - 15.0, center.1 - 9.0],
                Some(graphics::Color::BLACK),
            );
        }

        mb.build(ctx)
    }

    fn create_cell_mesh(
        &self,
        ctx: &mut Context,
        window_center: (f32, f32),
        r: f32,
        cell: &Cell,
        selected: bool,
    ) -> GameResult<(graphics::Mesh, (Piece, (f32, f32)))> {
        let pos = cell.position;
        let piece = cell.top_piece();
        let cell_center = position_to_point(pos, r, window_center);

        let mesh = self.create_hex_mesh(ctx, cell_center, r, Either::Left(piece), selected);

        mesh.map(|m| (m, (piece.clone(), cell_center)))
    }

    pub fn create_moveset_mesh(
        &mut self,
        ctx: &mut Context,
        player_color: Option<HiveColor>,
    ) -> GameResult<Vec<graphics::Mesh>> {
        if self.selected_piece.is_none() {
            return Ok(vec![]);
        }

        let window_center = self.window_center(ctx);
        let r = self.get_hex_radius(ctx);

        if self.game.hive.cells.is_empty() {
            let cell_center = position_to_point(0, r, window_center);

            let hex_mesh = self.create_hex_mesh(ctx, cell_center, r, Either::Right(""), false)?;
            return Ok(vec![hex_mesh]);
        }

        if player_color.is_none() {
            return Ok(vec![]);
        }
        let player_color = player_color.unwrap();

        if let Some(moveset) = &self.curr_moveset {
            let mut cell_meshes = Vec::with_capacity(self.game.hive.cells.len());

            for pos in moveset {
                let cell_center = position_to_point(*pos, r, window_center);
                let hex_mesh =
                    self.create_hex_mesh(ctx, cell_center, r, Either::Right(""), false)?;
                cell_meshes.push(hex_mesh);
            }

            return Ok(cell_meshes);
        }

        match self.selected_piece {
            Some(Either::Left(_)) => {
                let mut cell_meshes = Vec::with_capacity(self.game.hive.cells.len());

                let poss: Vec<_> = if self.game.hive.cells.len() == 1 {
                    self.game
                        .hive
                        .cell_neighboring_positions(0)
                        .into_iter()
                        .collect()
                } else {
                    self.game
                        .hive
                        .cells
                        .iter()
                        .filter(|&cell| cell.color() == player_color)
                        .flat_map(|cell| self.game.hive.cell_neighboring_positions(cell.position))
                        .filter(|&pos| {
                            !self
                                .game
                                .hive
                                .cell_neighboring_positions(pos)
                                .into_iter()
                                .any(|p| {
                                    self.game.hive.get_cell(p).map(Cell::color)
                                        == Some(!player_color)
                                })
                        })
                        .filter(|&pos| self.game.hive.get_cell(pos).is_none())
                        .collect()
                };

                for pos in &poss {
                    if self.game.hive.get_cell(*pos).is_none() {
                        let cell_center = position_to_point(*pos, r, window_center);
                        let hex_mesh =
                            self.create_hex_mesh(ctx, cell_center, r, Either::Right(""), false)?;
                        cell_meshes.push(hex_mesh);
                    }
                }

                self.curr_moveset = Some(poss);

                Ok(cell_meshes)
            }
            Some(Either::Right(pos)) => {
                let mut cell_meshes = Vec::with_capacity(self.game.hive.cells.len());

                let possible_paths = self.game.hive.possible_move_paths(pos);
                if !possible_paths.is_empty() {
                    trace!("possible paths: {:?}", possible_paths);
                }

                for path in &possible_paths {
                    let last_pos = *path.last().unwrap();
                    let cell_center = position_to_point(last_pos, r, window_center);
                    let hex_mesh =
                        self.create_hex_mesh(ctx, cell_center, r, Either::Right(""), false)?;
                    cell_meshes.push(hex_mesh);
                }

                self.curr_moveset =
                    Some(possible_paths.iter().map(|p| *p.last().unwrap()).collect());

                Ok(cell_meshes)
            }
            _ => unreachable!(),
        }
    }

    pub fn create_selection_mesh(&self, ctx: &mut Context) -> GameResult<Option<graphics::Mesh>> {
        if let Some(Either::Right(pos)) = self.selected_piece {
            let window_center = self.window_center(ctx);
            let r = self.get_hex_radius(ctx);

            if let Some(cell) = self
                .game
                .hive
                .cells
                .iter()
                .find(|cell| cell.position == pos)
            {
                let cell = &self
                    .game
                    .hive
                    .cells
                    .iter()
                    .find(|c| c.position == cell.position)
                    .unwrap();
                let cell_mesh = self.create_cell_mesh(ctx, window_center, r, cell, true)?;

                Ok(Some(cell_mesh.0))
            } else {
                Ok(None)
            }
        } else {
            Ok(None)
        }
    }

    pub fn create_hive_mesh(
        &self,
        ctx: &mut Context,
    ) -> GameResult<(Vec<graphics::Mesh>, Vec<(Piece, (f32, f32))>)> {
        let window_center = self.window_center(ctx);
        let r = self.get_hex_radius(ctx);

        let mut cell_meshes = Vec::with_capacity(self.game.hive.cells.len());
        let mut piece_positions = vec![];

        for cell in &self.game.hive.cells {
            let (cell_mesh, piece_position) =
                self.create_cell_mesh(ctx, window_center, r, cell, false)?;
            cell_meshes.push(cell_mesh);
            piece_positions.push(piece_position);
        }

        Ok((cell_meshes, piece_positions))
    }

    pub fn create_player_ui_meshes(
        &self,
        ctx: &mut Context,
    ) -> GameResult<(Vec<graphics::Mesh>, Vec<(Piece, (f32, f32))>)> {
        let r = R;
        let offset = r * 2.0;

        let mut piece_meshes =
            Vec::with_capacity(self.game.players.iter().map(|p| p.pieces.len()).sum());
        let mut ui_piece_positions = Vec::new();

        for player in &self.game.players {
            let y = if player.color == HiveColor::White {
                offset
            } else {
                self.window_dimensions(ctx).1 - offset
            };

            for (i, piece) in player.pieces.iter().enumerate() {
                let selected = if let Some(Either::Left((color, idx))) = &self.selected_piece {
                    (*color, *idx) == (player.color, i)
                } else {
                    false
                };

                let center = (offset + (i as f32 * r * 2.0), y);
                let piece_mesh =
                    self.create_hex_mesh(ctx, center, r, Either::Left(piece), selected)?;
                piece_meshes.push(piece_mesh);
                ui_piece_positions.push((piece.clone(), center));
            }

            let pass_mesh = self.create_hex_mesh(
                ctx,
                (self.window_dimensions(ctx).0 - offset, y),
                r,
                Either::Right("PASS"),
                false,
            )?;
            piece_meshes.push(pass_mesh);
        }

        Ok((piece_meshes, ui_piece_positions))
    }
}
