use super::{Color, GameError, Hive, MoveError, Piece, Player, Position};

#[derive(Debug)]
pub enum ActionKind {
    Movement(Position, Position),
    Placement(Piece, Position),
    Pass,
}

#[derive(Debug)]
pub struct Action {
    pub player_color: Color,
    pub kind: ActionKind,
}

#[derive(Debug, PartialEq, Eq)]
pub enum GameStatus {
    Draw,
    GameOver(Color),
    Ongoing,
}

#[derive(Debug)]
pub struct Game {
    pub hive: Hive,
    pub players: [Player; 2],
    pub history: Vec<Action>,
}

impl Default for Game {
    fn default() -> Self {
        Self {
            hive: Hive::default(),
            players: [Player::new(Color::White), Player::new(Color::Black)],
            history: vec![],
        }
    }
}

impl Game {
    pub fn status(&self) -> GameStatus {
        self.hive.status()
    }

    pub fn current_player(&self) -> Option<Color> {
        self.history
            .last()
            .map(|last_action| !last_action.player_color)
    }

    pub fn progress(&mut self, action: Action) -> Result<GameStatus, GameError> {
        // don't let a single player act twice in a row; even if the other player passed
        // their last move, this is still registered as their distinct game action
        if let Some(last_action) = self.history.last() {
            if last_action.player_color == action.player_color {
                return Err(GameError::DoubleMove);
            }
        }

        match &action.kind {
            ActionKind::Movement(start, end) => {
                if let Some(path) = self
                    .hive
                    .possible_move_paths(*start)
                    .into_iter()
                    .find(|path| path.last() == Some(end))
                {
                    self.hive.move_piece(&path)?;
                } else {
                    return Err(MoveError::InvalidTargetPosition.into());
                }
            }
            ActionKind::Placement(piece, pos) => {
                self.hive.place_piece(piece.clone(), *pos)?;
            }
            ActionKind::Pass => {}
        }

        self.history.push(action);

        Ok(self.status())
    }
}
