use std::collections::HashSet;

use itertools::Itertools;

use super::{
    hex::{
        are_grouped, are_neighbors, is_path_consistent, neighboring_positions,
        DirectionalNeighborIter,
    },
    Color, GameStatus, MoveError, Piece, PlacementError, Position, Species,
};

#[derive(Debug)]
pub struct Cell {
    // the position of the cell
    pub position: Position,
    // the pieces can be stacked on one another; the last one is the one on the top
    pub pieces: Vec<Piece>,
}

impl Cell {
    pub fn new(position: Position, piece: Piece) -> Self {
        Self {
            position,
            pieces: vec![piece],
        }
    }

    pub fn color(&self) -> Color {
        self.top_piece().color
    }

    pub fn top_piece(&self) -> &Piece {
        self.pieces.last().unwrap()
    }
}

struct OutlierNeighborIter<'a> {
    hive: &'a Hive,
    curr_pos: Position,
    prev_poss: Vec<Position>,
}

impl<'a> OutlierNeighborIter<'a> {
    fn new(hive: &'a Hive, starting_pos: Position, mut prev_poss: Vec<Position>) -> Self {
        prev_poss.push(starting_pos);
        Self {
            hive,
            curr_pos: starting_pos,
            prev_poss,
        }
    }
}

impl<'a> Iterator for OutlierNeighborIter<'a> {
    type Item = Position;

    fn next(&mut self) -> Option<Self::Item> {
        let next = self
            .hive
            .cell_neighboring_positions(self.curr_pos)
            .into_iter()
            .filter(|&p| !self.prev_poss.contains(&p) && self.hive.get_cell(p).is_none())
            .filter(|&p| {
                self.hive
                    .cell_neighbors(p)
                    .map(|c| c.position)
                    .filter(|&p| p != self.curr_pos)
                    .filter(|&p| {
                        self.hive
                            .cells
                            .iter()
                            .map(|c| c.position)
                            .filter(|p2| !self.prev_poss.contains(p2))
                            .any(|p2| p2 == p)
                    })
                    .count()
                    != 0
            })
            .max_by_key(|&seed| {
                OutlierNeighborIter::new(self.hive, seed, self.prev_poss.clone()).count()
            });

        if let Some(pos) = next {
            self.curr_pos = pos;
            self.prev_poss.push(pos);
        }

        next
    }
}

#[derive(Debug, Default)]
pub struct Hive {
    pub cells: Vec<Cell>,
}

impl Hive {
    fn add_cell(&mut self, pos: Position, piece: Piece) {
        self.cells.push(Cell::new(pos, piece));
    }

    pub fn cell_color(&self, pos: Position) -> Option<Color> {
        self.get_cell(pos).map(Cell::color)
    }

    fn remove_cell(&mut self, pos: Position) {
        let idx = self
            .cells
            .iter()
            .position(|cell| cell.position == pos)
            .unwrap();
        self.cells.remove(idx);
    }

    pub fn get_cell(&self, pos: Position) -> Option<&Cell> {
        self.cells.iter().find(|cell| cell.position == pos)
    }

    fn get_cell_mut(&mut self, pos: Position) -> Option<&mut Cell> {
        self.cells.iter_mut().find(|cell| cell.position == pos)
    }

    pub fn cell_neighboring_positions(&self, pos: Position) -> [Position; 6] {
        neighboring_positions(pos)
    }

    fn cell_neighbors<'a>(&'a self, pos: Position) -> impl Iterator<Item = &Cell> + 'a {
        let neighbor_positions = self.cell_neighboring_positions(pos);

        self.cells
            .iter()
            .filter(move |cell| neighbor_positions.contains(&cell.position))
    }

    fn all_pieces<'a>(&'a self) -> impl Iterator<Item = &Piece> + 'a {
        self.cells.iter().flat_map(|c| c.pieces.iter())
    }

    fn num_neighbors(&self, pos: Position) -> usize {
        self.cell_neighbors(pos).count()
    }

    pub fn has_freedom_to_move(
        &self,
        source: Position,
        target: Position,
        ignore: Option<Position>,
    ) -> bool {
        assert!(are_neighbors(source, target)); // should be run on neighbors

        let mut neighbors1 = self
            .cell_neighbors(source)
            .into_iter()
            .map(|cell| cell.position)
            .filter(|pos| Some(*pos) != ignore)
            .collect::<HashSet<_>>();
        neighbors1.remove(&target);
        let mut neighbors2 = self
            .cell_neighbors(target)
            .into_iter()
            .map(|cell| cell.position)
            .filter(|pos| Some(*pos) != ignore)
            .collect::<HashSet<_>>();
        neighbors2.remove(&source);

        neighbors1.intersection(&neighbors2).count() != 2
    }

    pub fn status(&self) -> GameStatus {
        let mut surrounded_queens = self.cells.iter().filter(|cell| {
            cell.pieces.iter().any(|p| p.species == Species::QueenBee)
                && self.num_neighbors(cell.position) == 6
        });

        if let Some(cell1) = surrounded_queens.next() {
            if let Some(_cell2) = surrounded_queens.next() {
                GameStatus::Draw
            } else {
                GameStatus::GameOver(cell1.color())
            }
        } else {
            GameStatus::Ongoing
        }
    }

    fn run_common_move_checks(
        &self,
        player_color: Color,
        path: &[Position],
    ) -> Result<(), MoveError> {
        if path.len() < 2 {
            return Err(MoveError::NoMovement);
        }

        if !self
            .cells
            .iter()
            .map(|cell| &cell.pieces)
            .any(|pcs| pcs.contains(&Piece::new(player_color, Species::QueenBee)))
        {
            return Err(MoveError::QueenBeeNotPlaced);
        }

        let cell = if let Some(cell) = self.get_cell(*path.first().unwrap()) {
            cell
        } else {
            return Err(MoveError::MissingPiece);
        };

        let piece = if let Some(piece) = cell.pieces.last() {
            piece
        } else {
            return Err(MoveError::MissingPiece);
        };

        if piece.color != player_color {
            return Err(MoveError::WrongPlayer);
        }

        if !is_path_consistent(path) {
            return Err(MoveError::Inconsistent);
        }

        let is_cell_flat = cell.pieces.len() == 1;

        // check if the one hive rule holds wrt the piece itself
        if self.cell_neighbors(path[0]).count() == 1 && {
            let mut next_move_neighbors = self
                .cell_neighbors(path[1])
                .filter(|c| c.position != path[0]);
            let first_neighbor = next_move_neighbors.next();
            if next_move_neighbors.next().is_some() {
                false
            } else if let Some(neighbor) = first_neighbor {
                !self
                    .cell_neighbors(path[0])
                    .any(|c| c.position == neighbor.position)
                    && self.cell_neighbors(neighbor.position).count() == 1
            } else {
                false
            }
        } {
            return Err(MoveError::OneHiveRuleViolation);
        }

        // check if the one hive rule holds wrt the rest of the hive
        if is_cell_flat {
            let potential_gap = path[0];

            let cell_positions = self
                .cells
                .iter()
                .map(|cell| cell.position)
                .filter(|&pos| pos != potential_gap)
                .collect::<Vec<_>>();

            if !are_grouped(&cell_positions) {
                return Err(MoveError::OneHiveRuleViolation);
            }
        }

        // check if the whole path neighbors at least one of the existing ones
        if path.iter().skip(1).any(|pos| {
            if is_cell_flat {
                self.cell_neighbors(*pos)
                    .filter(|&cell| cell.position != path[0])
                    .count()
                    == 0
            } else {
                self.num_neighbors(*pos) == 0
            }
        }) {
            return Err(MoveError::NotAdjacentToAnotherPiece);
        }

        Ok(())
    }

    pub fn move_piece(&mut self, path: &[Position]) -> Result<(), MoveError> {
        let piece = &self.get_cell(path[0]).unwrap().top_piece();

        self.run_common_move_checks(piece.color, path)?;

        // perform the actual move
        match piece.species {
            Species::Beetle => self.move_beetle(path),
            Species::Grasshopper => self.move_grasshopper(path),
            Species::QueenBee => self.move_queen_bee(path),
            Species::SoldierAnt => self.move_soldier_ant(path),
            Species::Spider => self.move_spider(path),
        }
    }

    pub fn can_move(&self, path: &[Position]) -> Result<(), MoveError> {
        let piece = &self.get_cell(path[0]).unwrap().top_piece();

        self.run_common_move_checks(piece.color, path)?;

        // perform the actual move
        match piece.species {
            Species::Beetle => self.can_beetle_move(path),
            Species::Grasshopper => self.can_grasshopper_move(path),
            Species::QueenBee => self.can_queen_bee_move(path),
            Species::SoldierAnt => self.can_soldier_ant_move(path),
            Species::Spider => self.can_spider_move(path),
        }
    }

    fn can_beetle_move(&self, path: &[Position]) -> Result<(), MoveError> {
        if path.len() != 2 || !are_neighbors(path[0], path[1]) {
            return Err(MoveError::TooLong);
        }

        if self.get_cell(path[1]).is_none() && !self.has_freedom_to_move(path[0], path[1], None) {
            return Err(MoveError::NoFreedomToMove);
        }

        Ok(())
    }

    fn move_beetle(&mut self, path: &[Position]) -> Result<(), MoveError> {
        self.can_beetle_move(path)?;

        let source_cell = self.get_cell_mut(path[0]).unwrap();
        let piece = source_cell.pieces.pop().unwrap();
        if source_cell.pieces.is_empty() {
            self.remove_cell(path[0]);
        }

        if let Some(target_cell) = self.get_cell_mut(path[1]) {
            target_cell.pieces.push(piece);
        } else {
            self.add_cell(path[1], piece);
        }

        Ok(())
    }

    fn can_grasshopper_move(&self, path: &[Position]) -> Result<(), MoveError> {
        if self.get_cell(*path.last().unwrap()).is_some() {
            return Err(MoveError::CantClimb);
        }

        if !(0..6)
            .map(|dir| {
                let mut path = vec![path[0]];
                let mut iter = DirectionalNeighborIter::new(path[0], dir);
                path.extend(iter.by_ref().take_while(|&np| self.get_cell(np).is_some()));
                path.push(iter.curr_pos());
                path
            })
            .filter(|p| p.len() >= 3)
            .any(|p| p == path)
        {
            return Err(MoveError::GrasshopperBadHop);
        }

        Ok(())
    }

    fn move_grasshopper(&mut self, path: &[Position]) -> Result<(), MoveError> {
        self.can_grasshopper_move(path)?;

        // instead of creating a target cell and removing the source one, just change the
        // position of the latter
        self.get_cell_mut(path[0]).unwrap().position = *path.last().unwrap();

        Ok(())
    }

    fn can_queen_bee_move(&self, path: &[Position]) -> Result<(), MoveError> {
        if path.len() != 2 || !are_neighbors(path[0], path[1]) {
            return Err(MoveError::TooLong);
        }

        if self.get_cell(path[1]).is_some() {
            return Err(MoveError::CantClimb);
        }

        if !self.has_freedom_to_move(path[0], path[1], None) {
            return Err(MoveError::NoFreedomToMove);
        }

        Ok(())
    }

    fn move_queen_bee(&mut self, path: &[Position]) -> Result<(), MoveError> {
        self.can_queen_bee_move(path)?;

        // instead of creating a target cell and removing the source one, just change the
        // position of the latter
        self.get_cell_mut(path[0]).unwrap().position = path[1];

        Ok(())
    }

    fn can_soldier_ant_move(&self, path: &[Position]) -> Result<(), MoveError> {
        if path.iter().skip(1).any(|pos| self.get_cell(*pos).is_some()) {
            return Err(MoveError::CantClimb);
        }

        if !are_grouped(path) {
            return Err(MoveError::Inconsistent);
        }

        for window in path.windows(2) {
            if let [pos1, pos2] = window {
                if !self.has_freedom_to_move(*pos1, *pos2, Some(path[0])) {
                    return Err(MoveError::NoFreedomToMove);
                }
            } else {
                unreachable!();
            }
        }

        Ok(())
    }

    fn move_soldier_ant(&mut self, path: &[Position]) -> Result<(), MoveError> {
        self.can_soldier_ant_move(path)?;

        // instead of creating a target cell and removing the source one, just change the
        // position of the latter
        self.get_cell_mut(path[0]).unwrap().position = *path.last().unwrap();

        Ok(())
    }

    fn can_spider_move(&self, path: &[Position]) -> Result<(), MoveError> {
        if path.len() != 4 {
            return Err(MoveError::SpiderNotThreeSpaces);
        }

        if path.iter().skip(1).any(|pos| self.get_cell(*pos).is_some()) {
            return Err(MoveError::CantClimb);
        }

        if !path.iter().all_unique() {
            return Err(MoveError::SpiderBacktracking);
        }

        if !are_grouped(path) {
            return Err(MoveError::Inconsistent);
        }

        for window in path.windows(2) {
            if let [pos1, pos2] = window {
                if !self.has_freedom_to_move(*pos1, *pos2, Some(path[0])) {
                    return Err(MoveError::NoFreedomToMove);
                }
            } else {
                unreachable!();
            }
        }

        Ok(())
    }

    fn move_spider(&mut self, path: &[Position]) -> Result<(), MoveError> {
        self.can_spider_move(path)?;

        // instead of creating a target cell and removing the source one, just change the
        // position of the latter
        self.get_cell_mut(path[0]).unwrap().position = *path.last().unwrap();

        Ok(())
    }

    pub fn place_piece(&mut self, piece: Piece, pos: Position) -> Result<(), PlacementError> {
        // a piece can't be placed on top of an existing piece (only moved there)
        if self.get_cell(pos).is_some() {
            return Err(PlacementError::Topping);
        }

        // unless it's the first piece in the game, it must be placed next to an existing one
        if !self.cells.is_empty() && self.num_neighbors(pos) == 0 {
            return Err(PlacementError::NotAdjacentToAnotherPiece);
        }

        // unless it's the second piece in the game, it can't be placed next to an enemy one
        if self.cells.len() > 1
            && self
                .cell_neighbors(pos)
                .any(|cell| cell.color() != piece.color)
        {
            return Err(PlacementError::EnemyNeighbors);
        }

        // each player must place the queen bee by the end of their 4th move
        if self
            .cells
            .iter()
            .flat_map(|c| c.pieces.iter().map(|p| p.color))
            .filter(|&c| c == piece.color)
            .count()
            == 3
            && !self
                .cells
                .iter()
                .map(|cell| &cell.pieces)
                .any(|pcs| pcs.contains(&Piece::new(piece.color, Species::QueenBee)))
            && piece.species != Species::QueenBee
        {
            return Err(PlacementError::MustPlaceQueenBee);
        }

        // create the cell with the piece on it
        self.add_cell(pos, piece);

        Ok(())
    }

    pub fn possible_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        assert!(self.get_cell(pos).is_some());

        match self.get_cell(pos).unwrap().top_piece().species {
            Species::Beetle => self.possible_beetle_move_paths(pos),
            Species::Grasshopper => self.possible_grasshopper_move_paths(pos),
            Species::QueenBee => self.possible_queen_bee_move_paths(pos),
            Species::SoldierAnt => self.possible_soldier_ant_move_paths(pos),
            Species::Spider => self.possible_spider_move_paths(pos),
        }
    }

    fn possible_beetle_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        self.cell_neighboring_positions(pos)
            .into_iter()
            .map(|dest| vec![pos, dest])
            .filter(|path| self.can_move(path).is_ok())
            .collect()
    }

    fn possible_grasshopper_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        (0..6)
            .map(|dir| {
                let mut path = vec![pos];
                let mut iter = DirectionalNeighborIter::new(pos, dir);
                path.extend(iter.by_ref().take_while(|&np| self.get_cell(np).is_some()));
                path.push(iter.curr_pos());
                path
            })
            .filter(|path| path.len() >= 3 && self.can_move(path).is_ok())
            .collect()
    }

    fn possible_queen_bee_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        self.cell_neighboring_positions(pos)
            .into_iter()
            .filter(|&p| self.get_cell(p).is_none())
            .map(|dest| vec![pos, dest])
            .filter(|path| self.can_move(path).is_ok())
            .collect()
    }

    fn possible_soldier_ant_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        let mut possible_paths: Vec<_> = self
            .cell_neighboring_positions(pos)
            .into_iter()
            .filter(|&p| self.get_cell(p).is_none())
            .filter(|&p| {
                self.cell_neighbors(p)
                    .filter(|&cell| cell.position != pos)
                    .count()
                    != 0
            })
            .map(|p| {
                let mut path = vec![pos, p];
                path.extend(OutlierNeighborIter::new(self, p, vec![pos, p]));
                path
            })
            .flat_map(|path| {
                (2..path.len())
                    .map(|i| path[..i].to_vec())
                    .collect::<Vec<_>>()
            })
            .filter(|path| self.can_move(path).is_ok())
            .collect();
        possible_paths.sort_unstable_by_key(|p| p.len());

        let mut end_poss = HashSet::new();

        possible_paths
            .into_iter()
            .filter(|p| end_poss.insert(*p.last().unwrap()))
            .collect()
    }

    fn possible_spider_move_paths(&self, pos: Position) -> Vec<Vec<Position>> {
        self.possible_soldier_ant_move_paths(pos)
            .into_iter()
            .filter(|p| p.len() == 4)
            .collect()
    }

    pub fn must_pass(&self, player_color: Color) -> bool {
        let placed_piece_count = self
            .all_pieces()
            .filter(|p| p.color == player_color)
            .count();

        if placed_piece_count == 0 {
            return false;
        }

        placed_piece_count < 11
            && self
                .cells
                .iter()
                .map(|c| c.top_piece().color)
                .all(|c| c == !player_color)
            || placed_piece_count == 11
                && self
                    .cells
                    .iter()
                    .filter(|c| c.top_piece().color == player_color)
                    .map(|c| c.position)
                    .all(|p| self.possible_move_paths(p).is_empty())
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn placement_initial_piece() {
        let mut hive = Hive::default();
        let first_piece = Piece::new(Color::White, Species::Spider);

        assert!(hive.cells.is_empty());

        hive.place_piece(first_piece, 0).unwrap();

        assert_eq!(hive.num_neighbors(0), 0);

        assert_eq!(hive.cells.len(), 1);
        assert_eq!(hive.status(), GameStatus::Ongoing);
    }

    #[test]
    fn placement_second_piece() {
        let mut hive = Hive::default();

        let piece1 = Piece::new(Color::White, Species::Spider);
        hive.place_piece(piece1, 0).unwrap();

        let piece2 = Piece::new(Color::Black, Species::Spider);
        hive.place_piece(piece2, 1).unwrap();

        assert_eq!(hive.num_neighbors(0), 1);
        assert_eq!(hive.num_neighbors(1), 1);
    }

    #[test]
    fn beetle_movement() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(2, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(4, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(5, Piece::new(Color::White, Species::Spider)),
                Cell::new(6, Piece::new(Color::White, Species::Beetle)),
                Cell::new(8, Piece::new(Color::Black, Species::Grasshopper)),
            ],
        };

        // valid moves
        hive.can_move(&[6, 0]).unwrap();
        hive.can_move(&[6, 1]).unwrap();
        hive.can_move(&[6, 5]).unwrap();
        hive.can_move(&[6, 16]).unwrap();

        // some invalid moves
        assert_eq!(
            hive.can_move(&[6, 17]),
            Err(MoveError::NotAdjacentToAnotherPiece)
        );
        assert_eq!(
            hive.can_move(&[6, 18]),
            Err(MoveError::NotAdjacentToAnotherPiece)
        );
    }

    #[test]
    fn grasshopper_movement() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::Spider)),
                Cell::new(2, Piece::new(Color::Black, Species::Beetle)),
                Cell::new(3, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(4, Piece::new(Color::Black, Species::Grasshopper)),
                Cell::new(6, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(9, Piece::new(Color::Black, Species::Spider)),
                Cell::new(13, Piece::new(Color::White, Species::Beetle)),
                Cell::new(14, Piece::new(Color::White, Species::Grasshopper)),
                Cell::new(15, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(16, Piece::new(Color::Black, Species::SoldierAnt)),
                Cell::new(23, Piece::new(Color::White, Species::Spider)),
                Cell::new(24, Piece::new(Color::Black, Species::Spider)),
            ],
        };

        // valid moves
        hive.can_move(&[14, 15, 32]).unwrap();
        hive.can_move(&[14, 13, 27]).unwrap();
        hive.can_move(&[14, 4, 3, 10]).unwrap();

        assert_eq!(hive.possible_grasshopper_move_paths(14).len(), 3);

        // some invalid moves
        assert_eq!(hive.can_move(&[14, 18]), Err(MoveError::GrasshopperBadHop));
        assert_eq!(
            hive.can_move(&[14, 55]),
            Err(MoveError::NotAdjacentToAnotherPiece)
        );
    }

    #[test]
    fn queen_bee_movement() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(1, Piece::new(Color::Black, Species::Beetle)),
                Cell::new(2, Piece::new(Color::White, Species::Beetle)),
                Cell::new(3, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(6, Piece::new(Color::Black, Species::Spider)),
                Cell::new(12, Piece::new(Color::White, Species::Grasshopper)),
                Cell::new(13, Piece::new(Color::Black, Species::Grasshopper)),
                Cell::new(14, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(15, Piece::new(Color::Black, Species::SoldierAnt)),
                Cell::new(16, Piece::new(Color::White, Species::Spider)),
            ],
        };

        // valid moves
        hive.can_move(&[14, 5]).unwrap();
        hive.can_move(&[14, 4]).unwrap();
        hive.can_move(&[14, 29]).unwrap();
        hive.can_move(&[14, 30]).unwrap();

        // some invalid moves
        assert_eq!(hive.can_move(&[14, 13]), Err(MoveError::CantClimb));
        assert_eq!(hive.can_move(&[14, 15]), Err(MoveError::CantClimb));
        assert_eq!(hive.can_move(&[14, 28]), Err(MoveError::TooLong));
        assert_eq!(hive.can_move(&[14, 31]), Err(MoveError::TooLong));
        assert_eq!(
            hive.can_move(&[14, 50]),
            Err(MoveError::NotAdjacentToAnotherPiece)
        );
        assert_eq!(
            hive.can_move(&[14, 51]),
            Err(MoveError::NotAdjacentToAnotherPiece)
        );
    }

    #[test]
    fn queen_bee_movement2() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(3, Piece::new(Color::Black, Species::QueenBee)),
            ],
        };

        // valid moves
        hive.can_move(&[0, 2]).unwrap();
        hive.can_move(&[0, 4]).unwrap();
        hive.can_move(&[3, 2]).unwrap();
        hive.can_move(&[3, 4]).unwrap();
    }

    #[test]
    fn soldier_ant_movement() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(3, Piece::new(Color::Black, Species::Beetle)),
                Cell::new(11, Piece::new(Color::Black, Species::SoldierAnt)),
                Cell::new(12, Piece::new(Color::White, Species::Beetle)),
                Cell::new(13, Piece::new(Color::White, Species::Grasshopper)),
                Cell::new(14, Piece::new(Color::Black, Species::QueenBee)),
            ],
        };

        // valid moves
        hive.can_move(&[11, 10]).unwrap();
        hive.can_move(&[11, 10, 2]).unwrap();
        hive.can_move(&[11, 10, 2, 1]).unwrap();
        hive.can_move(&[11, 10, 2, 1, 6]).unwrap();
        hive.can_move(&[11, 10, 2, 1, 6, 5]).unwrap();
        hive.can_move(&[11, 26]).unwrap();
        hive.can_move(&[11, 26, 27]).unwrap();
        hive.can_move(&[11, 26, 27, 28]).unwrap();
        hive.can_move(&[11, 26, 27, 28, 29]).unwrap();
        hive.can_move(&[11, 26, 27, 28, 29, 30]).unwrap();
        hive.can_move(&[11, 26, 27, 28, 29, 30, 15]).unwrap();

        assert_eq!(hive.possible_move_paths(11).len(), 11);

        // some invalid moves
        assert_eq!(
            hive.can_move(&[11, 10, 2, 1, 6, 5, 4]),
            Err(MoveError::NoFreedomToMove)
        );
    }

    #[test]
    fn soldier_ant_movement2() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(3, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(6, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(11, Piece::new(Color::Black, Species::SoldierAnt)),
            ],
        };

        assert_eq!(hive.possible_move_paths(6).len(), 9);
        assert_eq!(hive.possible_move_paths(11).len(), 9);
    }

    #[test]
    fn soldier_ant_movement3() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(3, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(10, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(12, Piece::new(Color::Black, Species::SoldierAnt)),
            ],
        };

        assert_eq!(hive.possible_move_paths(10).len(), 9);
        assert_eq!(hive.possible_move_paths(12).len(), 9);
    }

    #[test]
    fn spider_movement() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(3, Piece::new(Color::Black, Species::Beetle)),
                Cell::new(5, Piece::new(Color::White, Species::Spider)),
                Cell::new(11, Piece::new(Color::White, Species::SoldierAnt)),
                Cell::new(15, Piece::new(Color::Black, Species::Spider)),
                Cell::new(26, Piece::new(Color::Black, Species::Grasshopper)),
                Cell::new(28, Piece::new(Color::Black, Species::SoldierAnt)),
                Cell::new(29, Piece::new(Color::White, Species::Beetle)),
                Cell::new(47, Piece::new(Color::White, Species::Grasshopper)),
                Cell::new(48, Piece::new(Color::Black, Species::QueenBee)),
            ],
        };

        // valid moves
        hive.can_move(&[15, 16, 6, 1]).unwrap();
        hive.can_move(&[15, 14, 30, 51]).unwrap();
        hive.can_move(&[15, 14, 4, 12]).unwrap();
        hive.can_move(&[15, 14, 13, 27]).unwrap();

        // some invalid moves
        assert_eq!(
            hive.can_move(&[15, 16, 6, 1, 2]),
            Err(MoveError::SpiderNotThreeSpaces)
        );
        assert_eq!(
            hive.can_move(&[15, 16, 6, 16]),
            Err(MoveError::SpiderBacktracking)
        );
    }

    #[test]
    fn no_gap_jumping() {
        let hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(1, Piece::new(Color::White, Species::Spider)),
                Cell::new(3, Piece::new(Color::Black, Species::QueenBee)),
                Cell::new(9, Piece::new(Color::White, Species::Spider)),
                Cell::new(10, Piece::new(Color::Black, Species::Spider)),
            ],
        };

        hive.can_move(&[9, 2, 8, 7]).unwrap();
        hive.can_move(&[9, 8, 7, 18]).unwrap_err();
    }

    #[test]
    fn pass_obligation() {
        let mut hive = Hive {
            cells: vec![
                Cell::new(0, Piece::new(Color::White, Species::QueenBee)),
                Cell::new(1, Piece::new(Color::Black, Species::QueenBee)),
            ],
        };
        hive.cells[1]
            .pieces
            .push(Piece::new(Color::White, Species::Beetle));

        assert!(hive.must_pass(Color::Black));
    }
}
