use crate::pipelines::{start_web_server, WebServerState, WsActorMsg};
use crate::utils::{exec_cmd, get_file_hash};
use crate::{info, Args};
use anyhow::{anyhow, bail, Context, Result};
use futures::future::Future;
use log::error;
use notify::{event, RecommendedWatcher, RecursiveMode, Watcher};
use path_absolutize::Absolutize;
use std::path::{Path, PathBuf};
use tokio::fs::write;
use tokio::sync::watch;
use tokio::task;

pub const WEB_TARGET: &str = "wasm32-unknown-unknown";

/// web pipeline using wasm
pub struct WebPipeline {
    args: Args,
    /// hash of wasm file generated by cargo build
    wasm_hashed_name: String,
    project_name: String,
    target_dir: PathBuf,
}

impl WebPipeline {
    /// builds and generates Self struct
    pub async fn new(mut args: Args) -> Result<WebPipeline> {
        // init args
        {
            let web_args = args.sub_cmd.as_web_mut()?;
            // make output dir absolute
            web_args.output_dir = web_args
                .output_dir
                .absolutize()
                .with_context(|| {
                    format!(
                        "error getting absolute path for --output-dir {}",
                        web_args.output_dir.to_str().unwrap()
                    )
                })?
                .to_path_buf();
            // make output dir absolute
            if let Some(public_dir) = &mut web_args.public_dir {
                *public_dir = public_dir
                    .absolutize()
                    .with_context(|| {
                        format!(
                            "error getting absolute path for --public-dir {}",
                            public_dir.to_str().unwrap()
                        )
                    })?
                    .to_path_buf();
            }
        }
        let mut this = Self {
            args,
            //make dummy wasm_hashed_name
            wasm_hashed_name: String::new(),
            project_name: String::new(),
            target_dir: PathBuf::new(),
        };
        // build to make valid hashed_wasm_name
        this.build().await?;
        // get meta data
        {
            let mut cmd = cargo_metadata::MetadataCommand::new();
            cmd.manifest_path(this.args.project_dir.join("Cargo.toml"));
            let metadata = cmd.exec().unwrap();
            this.target_dir = PathBuf::from(metadata.target_directory.clone());
            this.project_name = metadata.root_package().unwrap().name.clone();
        }
        this.wasm_hashed_name = this.generate_wasm_hashed_name().await?;
        Ok(this)
    }

    /// runs commands according to args
    pub async fn run(this: Self) -> Result<()> {
        // check args
        {
            let web_args = &this.args.sub_cmd.as_web()?;
            // do a full build
            this.build_full().await?;
            const WATCHER_ERROR: &str = "Error while watching local file changes";
            const SERVER_ERROR: &str = "Error while launching server";

            // watch and serve
            if web_args.watch && web_args.serve.is_some() {
                // create channels only when hot reload is enabled
                let (build_tx, build_rx) = if web_args.hot_reload {
                    let (build_tx, build_rx) = watch::channel(WsActorMsg::None);
                    (Some(build_tx), Some(build_rx))
                } else {
                    (None, None)
                };

                let server = start_web_server(
                    WebServerState {
                        output_dir: web_args.output_dir.clone(),
                        public_dir: web_args.public_dir.clone(),
                        rx: build_rx,
                    },
                    web_args.serve.as_ref().unwrap().clone(),
                );
                let watcher = Self::watch(this, build_tx);

                // wait for both to complete and set context for each
                let (watcher_result, server_result) = tokio::join!(watcher, server);
                watcher_result.with_context(|| WATCHER_ERROR)??;
                server_result.with_context(|| SERVER_ERROR)??;
            }
            // if only serve
            else if let Some(serve) = &web_args.serve {
                start_web_server(
                    WebServerState {
                        output_dir: web_args.output_dir.clone(),
                        public_dir: web_args.public_dir.clone(),
                        rx: None,
                    },
                    serve.clone(),
                )
                .await
                .with_context(|| SERVER_ERROR)??;
            }
            // if only watch
            else if web_args.watch {
                Self::watch(this, None)
                    .await
                    .with_context(|| WATCHER_ERROR)??;
            }
        }
        Ok(())
    }

    pub fn watch(
        this: Self,
        build_tx: Option<watch::Sender<WsActorMsg>>,
    ) -> impl Future<Output = Result<Result<()>, task::JoinError>> {
        task::spawn(async move {
            info!("Watching");
            let (tx, mut rx) = watch::channel(());
            let mut watcher =
                RecommendedWatcher::new(move |res: notify::Result<event::Event>| match res {
                    Ok(event) => {
                        if let event::EventKind::Modify(event::ModifyKind::Data(_)) = &event.kind {
                            tx.send(()).unwrap();
                        }
                    }
                    Err(e) => error!("Error while watching dir\n{}", e),
                })
                .with_context(|| "Error initialising watcher")?;
            {
                let watch_path = this.args.project_dir.join("src");
                watcher
                    .watch(&watch_path, RecursiveMode::Recursive)
                    .with_context(|| {
                        format!("error watching {}/src", watch_path.to_str().unwrap())
                    })?;
            }
            while rx.changed().await.is_ok() {
                info!("Re-building");
                match this.build().await {
                    Err(err) => error!("Error while building\n{}", err),
                    // build full only when cargo build is successful
                    _ => {
                        this.build_full().await?;
                        if let Some(build_tx) = &build_tx {
                            build_tx.send(WsActorMsg::FileChange(this.wasm_hashed_name.clone()))?;
                        }
                    }
                }
            }

            Err::<(), anyhow::Error>(anyhow!("Watch exited unexpectedly"))
        })
    }

    /// builds bindings
    /// optimises build if release
    /// writes html
    pub async fn build_full(&self) -> Result<()> {
        let web_args = self.args.sub_cmd.as_web()?;
        // need not build again because it has already been done in new block
        // run wasm bindgen
        self.build_bindings().await?;
        // optimise bindings if release using binaryen
        if web_args.release {
            self.optimise_build().await?;
        }
        // write html
        write(
            Path::new(&web_args.output_dir).join("index.html"),
            self.generate_html(),
        )
        .await?;
        Ok(())
    }

    pub async fn optimise_build(&self) -> Result<()> {
        let web_sub_cmd = self.args.sub_cmd.as_web()?;
        let file_name = format!("{}.wasm", &self.wasm_hashed_name);
        exec_cmd(
            "wasm-opt",
            &["-Oz", "--dce", &file_name, "-o", &file_name],
            Some(&web_sub_cmd.output_dir),
            None,
        )
        .await?;
        Ok(())
    }

    /// generates path .wasm file generated by cargo build
    /// target/wasm32-unknown-unknown/{release/debug}/<name>.wasm
    pub fn generate_path_to_target_wasm(&self) -> Result<PathBuf> {
        let web_sub_cmd = self.args.sub_cmd.as_web()?;
        let path = self
            .target_dir
            .join(WEB_TARGET)
            .join(if web_sub_cmd.release {
                "release"
            } else {
                "debug"
            })
            .join(&format!("{}.wasm", &self.project_name));
        let path = path.absolutize()?;
        if !path.exists() {
            bail!("Expected wasm at {}", path.to_str().unwrap())
        }
        Ok(PathBuf::from(path))
    }

    /// generates hash for the wasm generated by cargo build
    /// prefixes index-
    pub async fn generate_wasm_hashed_name(&self) -> Result<String> {
        Ok(format!(
            "index-{}",
            get_file_hash(self.generate_path_to_target_wasm()?).await?
        ))
    }

    /// run cargo build
    pub async fn build(&self) -> Result<()> {
        let web_sub_cmd = self.args.sub_cmd.as_web()?;
        let mut args = vec!["build", "--target", WEB_TARGET];
        if web_sub_cmd.release {
            args.push("--release")
        }
        exec_cmd("cargo", &args, Some(&self.args.project_dir), None)
            .await
            .context("error running cargo to build for web")?;
        Ok(())
    }

    /// rust wasm-bindgen on the target binary
    pub async fn build_bindings(&self) -> Result<()> {
        let web_subcmd = self.args.sub_cmd.as_web()?;
        exec_cmd(
            "wasm-bindgen",
            &[
                self.generate_path_to_target_wasm()
                    .unwrap()
                    .to_str()
                    .unwrap(),
                // build for web
                "--target",
                "web",
                // no type script
                "--no-typescript",
                // dir to place the assets at
                "--out-dir",
                web_subcmd.output_dir.to_str().unwrap(),
                // name of output file
                "--out-name",
                &format!("{}.wasm", self.wasm_hashed_name.as_str()),
            ],
            Option::<&str>::None,
            None,
        )
            .await
            .context("error running cargo-bindgen. Make sure it is in path. Consider Installing it using `cargo install wasm-bindgen-cli`")?;
        Ok(())
    }

    /// generates html
    pub fn generate_html(&self) -> String {
        let web_args = self.args.sub_cmd.as_web().unwrap();
        let hot_reload_script = if web_args.hot_reload {
            format!(
                r#"(function () {{
    const socket = new WebSocket('ws://{serve_addrs}/__gxi__');
    socket.addEventListener('open', function (event) {{
        console.log("Gxib > Connected to Server: Hot Reload Enabled");
    }});
    socket.addEventListener('close', function (event) {{
        console.error("Gxib > Disconnected from server");
        if (confirm('Disconnected from server. Refresh ?'))
            location.reload()
    }});
    socket.addEventListener('message', event => {{
        const data = JSON.parse(event.data);
        if (data.event === "FileChange")
            location.reload()
        console.log('Gxib > Message from server ', data);
    }});
}})()"#,
                serve_addrs = web_args.serve.as_ref().unwrap()
            )
        } else {
            String::new()
        };
        format!(
            r#"<!DOCTYPE html>
<html>
  <head>
    <meta charset="utf-8">
    <link rel="preload" href="/{name}.wasm" as="fetch" type="application/wasm">
    <link rel="modulepreload" href="/{name}.js">
  </head>
  <body>
    <script type="module">
        {hot_reload_script}
        import init from '/{name}.js'; init('/{name}.wasm');
    </script>
  </body>
</html>"#,
            name = self.wasm_hashed_name,
            hot_reload_script = hot_reload_script
        )
    }
}
