use quote::{quote, ToTokens};
use syn::__private::TokenStream2;
use syn::parse::{Parse, ParseStream};
use syn::Result;

use crate::InitType;

/// Parser for the [gxi_c_macro macro](../gxi_c_macro/macro.gxi_c_macro.html).
pub struct TreeParser(pub TokenStream2);

impl Parse for TreeParser {
    /// TODO: update doc
    fn parse(input: ParseStream) -> Result<Self> {
        Ok(if input.is_empty() {
            TreeParser(TokenStream2::new())
        } else {
            // default init type is child
            TreeParser(TreeParser::custom_parse(
                input,
                InitType::Child,
                false,
                true,
            )?)
        })
    }
}

impl TreeParser {
    /// Parses the `for` block as defined in the [Looping Section][../gxi_c_macro/macro.gxi_c_macro.html#Looping] of the [gxi_c_macro macro](../gxi_c_macro/macro.gxi_c_macro.html).
    fn parse_for_block(
        input: ParseStream, init_type: &InitType, is_first_component: bool,
    ) -> Result<TokenStream2> {
        if input.parse::<syn::token::For>().is_ok() {
            // parse : for loop_variable in loop_data_source { }
            let loop_variable = input.parse::<syn::Expr>()?;
            input.parse::<syn::token::In>()?;
            let loop_data_source = input.parse::<syn::Expr>()?;
            // parse the block with InitType::Sibling
            let parsed_loop_block = {
                let block_content = syn::group::parse_braces(&input)?.content;
                Self::custom_parse(&block_content, InitType::Sibling, true, false)?
            };
            // concatenate
            Ok(quote! {
                // parent node which will hold all the nodes from the for loop
                let (node, ..) = init_member(node.clone(), #init_type, |this| Pure::new(this), #is_first_component);
                {
                    // this node will act as the child of pure block
                    // because there can be only one child but many siblings
                    // component inside the loop will be the sibling of this pure node
                    let (node, ..) = init_member(node.clone(), InitType::Child, |this| Pure::new(this), false);
                    // prev_sibling
                    let mut prev_sibling = node.clone();
                    for #loop_variable in #loop_data_source {
                        let node = prev_sibling.clone();
                        #parsed_loop_block
                        prev_sibling = node;
                    }
                    // drop any left in the tree
                    // because the for loop may run a little less than the previous run
                    *prev_sibling.as_ref().borrow_mut().as_node_mut().get_sibling_mut() = None;
                }
            })
        } else {
            Ok(TokenStream2::new())
        }
    }

    /// generates the block to correctly drop `Pure` component without violating mutable rules.
    fn get_pure_remove_block(pure_index: u32) -> TokenStream2 {
        quote! {{
            let mut node_borrow = node.as_ref().borrow_mut();
            let pure = node_borrow.as_node_mut().as_any_mut().downcast_mut::<Pure>().unwrap();
            if pure.pure_index != #pure_index {
                pure.pure_index = #pure_index;
                pure.child = None;
            }
        }}
    }

    /// Parses the `if` block as defined in the [Conditional Rendering Section][../gxi_c_macro/macro.gxi_c_macro.html#Conditional-Rendering] of the [gxi_c_macro macro](../gxi_c_macro/macro.gxi_c_macro.html).
    ///
    /// *internals*
    ///
    /// Each `if` block is wrapped inside a `Pure` component. The Pure node has a `pure_index` of 0. Each wing of the if block is given a `pure_index`.
    /// When a `branch` of the if block is true, its `pure_index` is compared to the `pure_index` of the parent. If it is true then it means that
    /// during the previous render this `branch` was true which means that the element in this `branch` is already initialized. Therefore it is not required
    /// to be initialized. If the pure_index of parent and the if branch is not the same then it means that during the previous render `branch` was not
    /// initialized therefore it needs to be initialized now.
    ///
    /// If an else branch is not provided then an else branch with a Pure node is appended.
    ///
    fn parse_condition_block(
        input: &ParseStream, init_type: &InitType, is_first_component: bool,
    ) -> Result<TokenStream2> {
        // check for if
        if input.parse::<syn::token::If>().is_ok() {
            let mut pure_index = 0;
            let mut if_logic: TokenStream2 = input.parse::<syn::Expr>()?.to_token_stream().into();
            // chain starts with if block
            let mut chain = quote! { if #if_logic };
            loop {
                pure_index += 1;
                // concatenate
                {
                    let parsed_block = {
                        let block = syn::group::parse_braces(&input)?.content;
                        TreeParser::custom_parse(&block, InitType::Child, true, false)?
                    };
                    let pure_remove_block = TreeParser::get_pure_remove_block(pure_index);
                    chain = quote! { #chain {
                        #pure_remove_block
                        #parsed_block
                    }};
                }
                // if there is no if_logic then if expression has ended
                if if_logic.is_empty() {
                    break;
                } else if input.parse::<syn::token::Else>().is_ok() {
                    chain = quote! { #chain else };
                    // reset if logic
                    if_logic = TokenStream2::new();
                    // check for nested if
                    if input.parse::<syn::token::If>().is_ok() {
                        // parse if_logic of nested if
                        if_logic = input.parse::<syn::Expr>()?.to_token_stream().into();
                        chain = quote! { #chain if #if_logic };
                    }
                } else {
                    // if no else block then add a custom one which when executes destroys any existing child
                    let pure_remove_block = TreeParser::get_pure_remove_block(pure_index + 1);
                    chain = quote! { #chain else #pure_remove_block };
                    break;
                }
            }

            Ok(quote! {
                let (node, ..) = init_member(node.clone(), #init_type, |this| Pure::new(this), #is_first_component);
                { #chain }
            })
        } else {
            Ok(TokenStream2::new())
        }
    }

    /// Parses the Component with its properties and its children recursively from the syntax defined by the [gxi_c_macro macro](../gxi_c_macro/macro.gxi_c_macro.html)
    fn parse_component(
        input: &ParseStream, init_type: &InitType, is_first_component: bool,
    ) -> Result<TokenStream2> {
        if let Ok(name) = input.parse::<syn::Path>() {
            let mut static_props = vec![];
            let mut dynamic_props = vec![];
            //parse properties enclosed in parenthesis
            if let Ok(syn::group::Parens { content, .. }) = syn::group::parse_parens(&input) {
                // loop till every thing inside parenthesis is parsed
                loop {
                    if let Ok(syn::ExprAssign { left, right, .. }) =
                        content.parse::<syn::ExprAssign>()
                    {
                        // push closure and literals to static_props and others to dynamic_props
                        match *right {
                            syn::Expr::Closure(closure) => {
                                let closure_body = closure.body;
                                let closure_args = closure.inputs;
                                static_props.push(quote! {{
                                        let state_clone = Rc::clone(&this);
                                        node.#left(move |#closure_args| Self::update(state_clone.clone(),#closure_body) );
                                    }});
                            }
                            syn::Expr::Lit(literal) => {
                                static_props.push(quote! { node.#left(#literal); })
                            }
                            _ => dynamic_props.push(quote! { node.#left(#right); }),
                        }
                        // if stream is empty then break
                        if content.is_empty() {
                            break;
                        } else {
                            // else expect a comma
                            content.parse::<syn::token::Comma>()?;
                        }
                    } else {
                        break;
                    }
                }
            }
            // create node initializer methods
            let component = {
                // block which calls prop setter functions of the node
                let prop_setter_block = {
                    // if there are no props then return an empty TokenStream
                    if dynamic_props.is_empty() && static_props.is_empty() {
                        TokenStream2::new()
                    } else {
                        let mut block = quote! {
                            let mut node = node.as_ref().borrow_mut();
                            let node = node.as_node_mut().as_any_mut().downcast_mut::<#name>().unwrap();
                        };
                        if !static_props.is_empty() {
                            block = quote! {
                                #block
                                if is_new {
                                    #(#static_props)*
                                }
                            };
                        }
                        if !dynamic_props.is_empty() {
                            block = quote! {
                                #block
                                #(#dynamic_props)*
                            };
                        }
                        quote! {{
                            #block
                        }}
                    }
                };

                // if init_type is child then get self_substitute
                quote! {
                    let node = {
                        let (node, is_new) = init_member(node.clone(), #init_type, |this| #name::new(this), #is_first_component);
                        #prop_setter_block
                        #name::render(node.clone());
                        node
                    };
                }
            };

            // parse children
            let children = if let syn::__private::Ok(syn::group::Brackets { content, .. }) =
                syn::group::parse_brackets(&input)
            {
                // if content is empty don't parse it
                if content.is_empty() {
                    TokenStream2::new()
                } else {
                    TreeParser::custom_parse(&content, InitType::Child, true, false)?
                }
            } else {
                TokenStream2::new()
            };

            // concatenate every thing
            Ok(quote! {
                #component
                {
                    #children
                }
                //#name::render(node.clone());
            })
        } else {
            Ok(TokenStream2::new())
        }
    }

    /// Parses the `#children` statement as defined in the [#children statement section][../gxi_c_macro/macro.gxi_c_macro.html#children-statement] of the [gxi_c_macro macro](../gxi_c_macro/macro.gxi_c_macro.html).
    fn parse_child_injection(
        input: ParseStream, init_type: &InitType, is_first_component: bool,
    ) -> Result<TokenStream2> {
        if let Ok(_) = input.parse::<syn::token::Pound>() {
            let ident = input.parse::<syn::Ident>()?;
            return match &ident.to_string()[..] {
                "children" => Ok(quote! {
                    let node = {
                        let (node, ..) = init_member(node.clone(), #init_type, |this| Pure::new(this), #is_first_component);
                        {
                            let mut this_borrow = this.as_ref().borrow_mut();
                            match this_borrow.deref_mut() {
                                GxiNodeType::Component(t) => *t.get_self_substitute_mut() = Some(Rc::downgrade(&node)),
                                _ => unreachable!(),
                            }
                        }
                        node
                    };
                }),
                _ => Err(syn::Error::new(
                    ident.span().unwrap().into(),
                    "Expected children here",
                )),
            };
        } else {
            Ok(TokenStream2::new())
        }
    }

    /// anything inside a {} is copied and executed on every render call
    fn parse_execution_block(input: ParseStream) -> Result<TokenStream2> {
        if let Ok(b) = input.parse::<syn::Block>() {
            Ok(quote! {{ #b }})
        } else {
            Ok(TokenStream2::new())
        }
    }

    fn custom_parse(
        input: ParseStream, mut init_type: InitType, can_have_more_than_one_root_node: bool,
        mut is_first_component: bool,
    ) -> Result<TokenStream2> {
        let mut tree = TokenStream2::new();
        let mut has_one_root_component = false;
        loop {
            // check if input is empty
            if input.is_empty() {
                return Ok(tree);
            }
            // match and parse
            let parsed = {
                let execution_block = TreeParser::parse_execution_block(&input)?;
                if execution_block.is_empty() {
                    // for, conditional and execution_block functions produce a component
                    // if the tree already has a root component and it can't have more than one then throw error
                    if !can_have_more_than_one_root_node && has_one_root_component {
                        return Err(syn::Error::new(
                            input.span().unwrap().into(),
                            "didn't expect this here. Help: You can't have more than one node here.",
                        ));
                    }
                    let component_block =
                        TreeParser::parse_component(&input, &init_type, is_first_component)?;
                    let parsed = if component_block.is_empty() {
                        let conditional_block = TreeParser::parse_condition_block(
                            &input,
                            &init_type,
                            is_first_component,
                        )?;
                        if conditional_block.is_empty() {
                            let child_injection = TreeParser::parse_child_injection(
                                &input,
                                &init_type,
                                is_first_component,
                            )?;
                            if child_injection.is_empty() {
                                let for_parse = TreeParser::parse_for_block(
                                    &input,
                                    &init_type,
                                    is_first_component,
                                )?;
                                if for_parse.is_empty() {
                                    return Err(syn::Error::new(
                                        input.span().unwrap().into(),
                                        "didn't expect this here",
                                    ));
                                }
                                for_parse
                            } else {
                                child_injection
                            }
                        } else {
                            conditional_block
                        }
                    } else {
                        component_block
                    };
                    // now no longer any component can be root
                    is_first_component = false;
                    // there can only be one root component
                    has_one_root_component = true;
                    // there can only be one child, therefore after parsing a component
                    // it is guaranteed that the next component will be sibling
                    init_type = InitType::Sibling;
                    parsed
                } else {
                    execution_block
                }
            };

            tree = quote! { #tree #parsed };
            // there has to be a comma if input is not empty and the previous parse was successful
            if !input.is_empty() {
                input.parse::<syn::token::Comma>()?;
            }
        }
    }
}
