use serde::{Serialize, Deserialize};
use std::borrow::Cow;
use std::collections::HashMap;
use std::fmt::{self, Display};
use escaper::{encode_attribute, encode_minimal};

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Interface<'a> {
    /// The main widget.
    pub interface: Widget<'a>,
    /// The translation domain.
    pub domain: Option<Cow<'a, str>>,
}

impl Display for Interface<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, r#"<?xml version="1.0", encoding="UTF-8"?>"#)?;
        match &self.domain {
            Some(domain) => write!(f, r#"<interface domain="{}">"#, encode_attribute(&domain))?,
            None => write!(f, r#"<interface>"#)?,
        }
        write!(f, "{}", self.interface)?;
        write!(f, "</interface>")
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Widget<'a> {
    /// The widget's class, ie `GtkButton` or `AdwHeaderBar`
    pub class: Cow<'a, str>,
    /// The widget's id
    pub id: Option<Cow<'a, str>>,
    /// All the properties of the widget. Since they are serde flattened
    /// you don't have to include a `properties` value.
    #[serde(flatten)]
    pub properties: HashMap<Cow<'a, str>, Property<'a>>,
    /// All the children of the widget.
    #[serde(default)]
    pub children: Vec<Child<'a>>,
    /// All the pango attributes of widget. For labels.
    #[serde(default)]
    pub attributes: HashMap<Cow<'a, str>, Cow<'a, str>>,
    /// The styles of the widget.
    #[serde(default)]
    pub style: Cow<'a, [Cow<'a, str>]>,
    /// The widget's layout properties for grids.
    #[serde(default)]
    pub layout: HashMap<Cow<'a, str>, Property<'a>>,
    /// Any addition xml you wish to use.
    #[serde(default)]
    pub xml: Option<Cow<'a, str>>,
}

impl Display for Widget<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, r#"<object class="{}""#, encode_attribute(&self.class))?;
        if let Some(id) = &self.id {
            write!(f, r#" id="{}""#, encode_attribute(&id))?;
        }
        write!(f, ">")?;

        for (key, property) in &self.properties {
            property.format_as_xml_tag(key, f)?;
        }

        if !self.style.is_empty() {
            write!(f, "<style>")?;
            for style in &*self.style {
                write!(f, r#"<class name="{}"/>"#, encode_attribute(style))?;
            }
            write!(f, "</style>")?;
        }

        if !self.attributes.is_empty() {
            write!(f, "<attributes>")?;
            for (attr, val) in &self.attributes {
                write!(
                    f,
                    r#"<attribute name="{}" value="{}"/>"#,
                    encode_attribute(attr),
                    encode_attribute(val),
                )?;
            }
            write!(f, "</attributes>")?;
        }

        if !self.layout.is_empty() {
            write!(f, "<layout>")?;
            for (key, property) in &self.layout {
                write!(
                    f,
                    r#"<property name="{}">{}</property>"#,
                    encode_attribute(key),
                    property
                )?;
            }
            write!(f, "</layout>")?;
        }

        for child in &*self.children {
            write!(f, "{}", child)?;
        }

        if let Some(xml) = &self.xml {
            write!(f, "{}", xml)?;
        }
        
        write!(f, "</object>")
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq)]
#[serde(untagged)]
pub enum Property<'a> {
    /// A string. If the string needs to be translatable it should be
    /// prefixed with a `\_`. If, for whatever reason you need a string to
    /// start with this particular sequence begin it with a `\\_` instead.
    ///
    /// These translatable strings are the one case where I think XML is
    /// more suited, but alas it is worth it.
    String(Cow<'a, str>),
    /// A simple boolean value
    Bool(bool),
    /// A simple integer value
    Integer(usize),
    /// A float
    Float(f64),
    /// A widget
    Widget(Widget<'a>),
}

impl Property<'_> {
    /// Converts this into an xml tag
    pub fn format_as_xml_tag(
        &self,
        name: &str,
        f: &mut fmt::Formatter<'_>
    ) -> fmt::Result {
        write!(f, r#"<property name="{}""#, encode_attribute(name))?;
        match self {
            Self::String(string) => {
                // Check for translated string
                if string.starts_with(r#"\_"#) {
                    write!(
                        f,
                        r#"translatable="yes">{}"#,
                        encode_minimal(&string[2..])
                    )?;
                } else if string.starts_with(r#"\\_"#) {
                    write!(
                        f,
                        r#">{}"#,
                        encode_minimal(&string[3..])
                    )?;
                } else {
                    write!(
                        f,
                        r#">{}"#,
                        encode_minimal(&string)
                    )?;
                }
            }
            Self::Bool(boolean) => write!(f, ">{}", boolean)?,
            Self::Integer(integer) => write!(f, ">{}", integer)?,
            Self::Float(float) => write!(f, ">{}", float)?,
            Self::Widget(widget) => write!(f, ">{}", widget)?,
        }
        write!(f, "</property>")
    }
}

impl Display for Property<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::String(string) => write!(f, "{}", encode_minimal(string)),
            Self::Bool(boolean) => write!(f, "{}", boolean),
            Self::Integer(integer) => write!(f, "{}", integer),
            Self::Float(float) => write!(f, "{}", float),
            Self::Widget(widget) => write!(f, "{}", widget),
        }
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Child<'a> {
    pub child_type: Option<Cow<'a, str>>,
    #[serde(flatten)]
    pub widget: Widget<'a>
}

impl Display for Child<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "<child ")?;
        if let Some(child_type) = &self.child_type {
            write!(f, r#"type="{}""#, encode_attribute(&child_type))?;
        }
        write!(f, ">")?;

        write!(f, "{}", self.widget)?;

        write!(f, "</child>")
    }
}
