use serde::{Serialize, Deserialize};
use std::borrow::Cow;
use std::collections::HashMap;
use std::fmt::{self, Display};
use escaper::{encode_attribute, encode_minimal};

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Interface<'a> {
    pub interface: Widget<'a>
}

impl Display for Interface<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, r#"<?xml version="1.0", encoding="UTF-8"?><interface>{}</interface>"#, self.interface)
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Widget<'a> {
    pub class: Cow<'a, str>,
    pub id: Option<Cow<'a, str>>,
    #[serde(flatten)]
    pub properties: HashMap<Cow<'a, str>, Property<'a>>,
    #[serde(default)]
    pub children: Vec<Child<'a>>,
    #[serde(default)]
    pub attributes: HashMap<Cow<'a, str>, Cow<'a, str>>,
    #[serde(default)]
    pub style: Cow<'a, [Cow<'a, str>]>,
}

impl Display for Widget<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, r#"<object class="{}""#, encode_attribute(&self.class))?;
        if let Some(id) = &self.id {
            write!(f, r#" id="{}""#, encode_attribute(&id))?;
        }
        write!(f, ">")?;

        for (key, property) in &self.properties {
            write!(
                f,
                r#"<property name="{}">{}</property>"#,
                encode_attribute(key),
                property
            )?;
        }

        if !self.style.is_empty() {
            write!(f, "<style>")?;
            for style in &*self.style {
                write!(f, r#"<class name={}/>"#, encode_attribute(style))?;
            }
            write!(f, "</style>")?;
        }

        if !self.attributes.is_empty() {
            write!(f, "<attributes>")?;
            for (attr, val) in &self.attributes {
                write!(
                    f,
                    r#"<attribute name="{}" value={}/>"#,
                    encode_attribute(attr),
                    encode_attribute(val),
                )?;
            }
            write!(f, "</attributes>")?;
        }

        for child in &*self.children {
            write!(f, "{}", child)?;
        }
        
        write!(f, "</object>")
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq)]
#[serde(untagged)]
pub enum Property<'a> {
    String(Cow<'a, str>),
    Bool(bool),
    Integer(usize),
    Float(f64),
    Widget(Widget<'a>),
}

impl Display for Property<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::String(string) => write!(f, "{}", encode_minimal(string)),
            Self::Bool(boolean) => write!(f, "{}", boolean),
            Self::Integer(integer) => write!(f, "{}", integer),
            Self::Float(float) => write!(f, "{}", float),
            Self::Widget(widget) => write!(f, "{}", widget),
        }
    }
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Default)]
pub struct Child<'a> {
    pub child_type: Option<Cow<'a, str>>,
    #[serde(flatten)]
    pub widget: Widget<'a>
}

impl Display for Child<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "<child ")?;
        if let Some(child_type) = &self.child_type {
            write!(f, r#"type="{}""#, encode_attribute(&child_type))?;
        }
        write!(f, ">")?;

        write!(f, "{}", self.widget)?;

        write!(f, "</child>")
    }
}
