use nom::branch::alt;
use nom::bytes::complete::tag;
use nom::bytes::complete::take_till1;
use nom::character::complete::alpha1;
use nom::character::complete::alphanumeric1;
use nom::character::complete::char;
use nom::character::complete::line_ending;
use nom::character::complete::multispace1;
use nom::character::complete::not_line_ending;
use nom::character::complete::space0;
use nom::character::complete::space1;
use nom::combinator::all_consuming;
use nom::combinator::map;
use nom::combinator::not;
use nom::combinator::peek;
use nom::combinator::recognize;
use nom::multi::many0;
use nom::sequence::delimited;
use nom::sequence::pair;
use nom::sequence::preceded;
use nom::sequence::terminated;
use nom::sequence::tuple;
use nom::IResult;
use std::collections::HashMap;

/// Comments are prefixed with the # symbol.
pub fn comment(input: &str) -> IResult<&str, &str> {
    preceded(char('#'), not_line_ending)(input)
}

pub fn blank_lines(input: &str) -> IResult<&str, &str> {
    recognize(many0(alt((multispace1, comment))))(input)
}

fn identifier(input: &str) -> IResult<&str, &str> {
    recognize(pair(
        alt((alpha1, tag("_"))),
        many0(alt((alphanumeric1, tag("_")))),
    ))(input)
}

/// A file is a set of sections
pub fn parse_file(input: &str) -> IResult<&str, HashMap<String, Vec<Vec<&str>>>> {
    use std::collections::hash_map::Entry;

    map(all_consuming(many0(section)), |sections: Vec<_>| {
        let mut map: HashMap<String, Vec<Vec<&str>>> = HashMap::new();
        for (name, lines) in sections {
            match map.entry(name.to_string()) {
                Entry::Occupied(mut x) => {
                    x.get_mut().extend(lines);
                }
                Entry::Vacant(x) => {
                    x.insert(lines);
                }
            }
        }
        map
    })(input)
}

/// Sections are lines between section...end
fn section(input: &str) -> IResult<&str, (&str, Vec<Vec<&str>>)> {
    terminated(
        tuple((
            delimited(blank_lines, section_name, blank_lines),
            many0(section_line),
        )),
        terminated(section_end, blank_lines),
    )(input)
}

fn section_name(input: &str) -> IResult<&str, &str> {
    terminated(identifier, terminated(space0, line_ending))(input)
}

/// the end string are case-sensitive.
fn section_end(input: &str) -> IResult<&str, &str> {
    tag("end")(input)
}

/// each line is a set of columns
fn section_line(input: &str) -> IResult<&str, Vec<&str>> {
    terminated(preceded(not(peek(section_end)), columns), blank_lines)(input)
}

/// Data values can be comma- or space-delimited: commas are replaced with spaces when the game reads a line from the file.
fn columns(input: &str) -> IResult<&str, Vec<&str>> {
    many0(delimited(column_space, column, column_space))(input)
}

fn column(input: &str) -> IResult<&str, &str> {
    take_till1(|c: char| !c.is_ascii_graphic() || c == '#' || c == ',')(input)
}

pub fn column_space(input: &str) -> IResult<&str, &str> {
    recognize(many0(alt((space1, comment, tag(",")))))(input)
}

pub fn parse(input: &str) -> Option<std::collections::HashMap<String, Vec<Vec<&str>>>> {
    match parse_file(input) {
        Ok((_, v)) => Some(v),
        Err(_) => None,
    }
}

#[cfg(test)]
mod tests {
    use crate::parse;

    #[test]
    fn test1() {
        let parsed = parse(
            r#"

        objs
        1
        #test
        end
        objs
        2
        end

        #emptysection
        empty
        end

        cars
        test#comment
        end

        objs
        3   4        5
        end

        "#,
        );

        assert_eq!(parsed.is_some(), true);

        let content = parsed.unwrap();

        let objs = content.get("objs").unwrap();
        assert_eq!(objs.len(), 3);

        assert_eq!(objs[0][0], "1");
        assert_eq!(objs[1][0], "2");
        assert_eq!(objs[2][0], "3");
        assert_eq!(objs[2][1], "4");
        assert_eq!(objs[2][2], "5");

        let cars = content.get("cars").unwrap();
        assert_eq!(cars.len(), 1);
        assert_eq!(cars[0][0], "test");
    }
}
