use nom::branch::alt;
use nom::bytes::complete::tag;
use nom::bytes::complete::take_till1;
use nom::character::complete::alpha1;
use nom::character::complete::alphanumeric1;
use nom::character::complete::char;
use nom::character::complete::line_ending;
use nom::character::complete::multispace1;
use nom::character::complete::not_line_ending;
use nom::character::complete::space0;
use nom::character::complete::space1;
use nom::combinator::all_consuming;
use nom::combinator::map;
use nom::combinator::not;
use nom::combinator::peek;
use nom::combinator::recognize;
use nom::multi::many0;
use nom::sequence::delimited;
use nom::sequence::pair;
use nom::sequence::preceded;
use nom::sequence::terminated;
use nom::sequence::tuple;

use nom_locate::LocatedSpan;

pub type Span<'a> = LocatedSpan<&'a str>;
pub type R<'a, T> = nom::IResult<Span<'a>, T>;

#[derive(Debug, PartialEq)]
pub struct Section {
    pub name: String,
    pub lines: Vec<Vec<String>>,
}

/// Comments are prefixed with the # symbol.
pub fn comment(input: Span) -> R<Span> {
    preceded(char('#'), not_line_ending)(input)
}

pub fn blank_lines(input: Span) -> R<Span> {
    recognize(many0(alt((multispace1, comment))))(input)
}

fn identifier(input: Span) -> R<Span> {
    recognize(pair(
        alt((alpha1, tag("_"))),
        many0(alt((alphanumeric1, tag("_")))),
    ))(input)
}

/// A file is a set of sections
pub fn parse_file(input: Span) -> R<Vec<Section>> {
    all_consuming(many0(section))(input)
}

/// Sections are lines between section...end
fn section(input: Span) -> R<Section> {
    map(
        terminated(
            tuple((section_name, many0(section_line))),
            terminated(section_end, blank_lines),
        ),
        |(name, lines)| Section {
            name: name.to_string(),
            lines,
        },
    )(input)
}

fn section_name(input: Span) -> R<Span> {
    delimited(blank_lines, identifier, terminated(space0, line_ending))(input)
}

/// the end string are case-sensitive.
fn section_end(input: Span) -> R<Span> {
    tag("end")(input)
}

/// each line is a set of columns
fn section_line(input: Span) -> R<Vec<String>> {
    delimited(
        blank_lines,
        preceded(not(peek(section_end)), columns),
        blank_lines,
    )(input)
}

/// Data values can be comma- or space-delimited: commas are replaced with spaces when the game reads a line from the file.
fn columns(input: Span) -> R<Vec<String>> {
    many0(delimited(column_space, column, column_space))(input)
}

fn column(input: Span) -> R<String> {
    map(
        take_till1(|c: char| !c.is_ascii_graphic() || c == '#' || c == ','),
        |s: Span| s.to_string(),
    )(input)
}

pub fn column_space(input: Span) -> R<Span> {
    recognize(many0(alt((space1, comment, tag(",")))))(input)
}

pub fn parse(input: &str) -> Option<Vec<Section>> {
    match parse_file(Span::from(input)) {
        Ok((_, v)) => Some(v),
        Err(_) => None,
    }
}

#[cfg(test)]
mod tests {
    use crate::{parse, Section};

    #[test]
    fn it_works() {
        let input = r#"objs
        # comment
        123 456
        end"#;
        assert_eq!(
            parse(input),
            Some(vec![Section {
                name: String::from("objs"),
                lines: vec![vec![String::from("123"), String::from("456")]]
            }])
        );
    }
}
