# gstore

[![pipeline status](https://gitlab.com/loers/gstore/badges/main/pipeline.svg)](https://gitlab.com/loers/gstore/-/commits/main)
[![API](https://docs.rs/gstore/badge.svg)](https://docs.rs/gstore)

Global and local state management for GTK apps which implements the flux pattern. This crate targets apps written with [gtk-rust-app](https://gitlab.com/loers/gtk-rust-app) framework.

A State can be any kind of data structure an application is based on. For a 'counter' app it
might be a struct with a single u32 field.
Actions are enums which represent the possible features of the app affecting the state.


## Global store usage

```rust
// src/store.rs

pub const INCREMENT: &str = "increment";

// Given you have a global state
#[derive(Debug, Default, Clone, PartialEq, Eq)]
pub struct State {
    pub count: u128
}

// Using the store macro a 'store' for this state
store!(State);
// The following public interface will be generated:
    pub type Store = gstore::Store<State>
    /// Initialize the store and global state
    /// 
    /// # Arguments
    /// - initializer: Initialize the state.
    /// - reducer: Handle actions and mutate the state
    /// - middlewares: Pre- and post-reduce handlers
    pub fn init_store(
        initializer: impl Fn(&mut State), 
        reducer: impl Fn(gstore::Action, &mut State), 
        middleware: Vec<Box<dyn gstore::Middleware<State>>>
    )
    /// Get a static reference to the store
    pub fn store() -> &'static Store
```

```rust
// main.rs

mod store;

// Initialize the store
pub fn main() {
    init_store(
        |state| {
            state.count = 0;
        },
        |action, state| {
            match action.name() {
                crate::store::INCREMENT => {
                    let amount: u128 = action.arg().unwrap();
                    state.count += amount;
                }
            }
        },
        vec![
            MyMiddleware::new()
        ]
    )
}

// Implement a middleware
struct MyMiddleware;
impl MyMiddleware {
    pub fn new() -> Box<Self> {
        Box::new(MyMiddleware)
    }
}
impl gstore::Middleware<State> for MyMiddleware {
    fn pre_reduce(action: &Action, state: &State) {
        //...
    }

    fn post_reduce(action: &Action, state: &State) {
        //...
    }
}

// Then in your GTK app you can dispatch actions to the store

let button: gtk::Button = gtk::Button::builder().build();
button.connect_clicked(|b| {
    b.dispatch(crate::store::INCREMENT, 2)
});

```

## Performance data

gstore can print basic performance information via the environment variabl `GSTORE_PERF=1`;

## License

gstore is distributed under the terms of the GPL-3.0 license. See LICENSE for details.
