// SPDX-License-Identifier: GPL-3.0-or-later

use gstore::{self, dispatch, select, store};

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct State {
    current_page: String,
    page_history: Vec<String>,
    selection: (u32, u32),
}

impl Default for State {
    fn default() -> Self {
        Self {
            current_page: "home".to_string(),
            page_history: vec![],
            selection: Default::default(),
        }
    }
}

mod dispatch_during_reduce_does_not_work {
    use super::*;

    store!(State);

    static mut REDUCER_CALLS: Vec<String> = Vec::new();
    static mut SELECTOR_CALLS: Vec<&'static str> = Vec::new();

    #[test]
    fn test() {
        env_logger::try_init().ok();

        init_store(
            Default::default(),
            |action, state| {
                println!("REDUCE {}", action);
                unsafe { REDUCER_CALLS.push(action.name().into()) }
                match action.name() {
                    "navigate" => {
                        let page: String = action.arg().unwrap();
                        state.current_page = page;
                        dispatch!("navigated", state.current_page.clone());
                    }
                    "navigated" => {
                        let page: String = action.arg().unwrap();
                        state.page_history.push(page);
                    }
                    "other_action" => {
                        state.selection = (1, 1);
                    }
                    _ => {}
                }
                println!("REDUCED {:?}", state);
            },
            Vec::new(),
        );

        select!(|state| state.current_page => |state| {
            println!("select current_page");
            let current_page = &state.current_page;
            unsafe {
                if SELECTOR_CALLS.len() == 0 || SELECTOR_CALLS.len() == 1 {
                    assert_eq!(current_page, "home");
                } else {
                    assert_eq!(current_page, "settings");
                }
                SELECTOR_CALLS.push("select current page");
            }
        });

        select!(|state| state.page_history => |state| {
            let page_history = &state.page_history;
            unsafe {
                if SELECTOR_CALLS.len() == 0 || SELECTOR_CALLS.len() == 1 {
                    let a: Vec<String> = Vec::new();
                    assert_eq!(page_history, &a);
                } else {
                    assert_eq!(page_history, &vec!["home".to_string()]);
                }
                SELECTOR_CALLS.push("select page history");
            }
        });

        println!("Send dispatches");
        dispatch!("navigate", "settings");
        dispatch!("other_action");

        unsafe {
            assert_eq!(
                SELECTOR_CALLS,
                vec![
                    "select current page",
                    "select page history",
                    "select current page"
                ]
            );

            assert_eq!(
                REDUCER_CALLS,
                vec![
                    // gstore calls init on itself
                    "init".to_string(),
                    // first dispatch from above
                    "navigate".to_string(),
                    // the second dispatch
                    "other_action".to_string()
                ]
            );
        }
    }
}

mod dispatch_during_reduce_does_work_with_middlewares {
    use super::*;

    store!(State);

    static mut REDUCER_CALLS: Vec<String> = Vec::new();
    static mut SELECTOR_CALLS: Vec<&'static str> = Vec::new();

    struct NavigationMiddlware;
    impl NavigationMiddlware {
        pub fn new() -> Box<Self> {
            Box::new(NavigationMiddlware)
        }
    }
    impl gstore::Middleware<State> for NavigationMiddlware {
        fn post_reduce(&self, action: &gstore::Action, state: &State) {
            match action.name() {
                "navigate" => {
                    dispatch!("navigated", state.current_page.clone())
                }
                _ => {}
            }
        }
        fn pre_reduce(&self, action: &gstore::Action, _state: &State) {
            match action.name() {
                "navigate" => {
                    // check dispatch works in pre-reduce (not required for the 'navigation' topic)
                    dispatch!("other_action");
                }
                _ => {}
            }
        }
    }

    #[test]
    fn test() {
        env_logger::try_init().ok();

        init_store(
            Default::default(),
            |action, state| {
                println!("REDUCE {}", action);
                unsafe { REDUCER_CALLS.push(action.name().into()) }
                match action.name() {
                    "navigate" => {
                        let page: String = action.arg().unwrap();
                        state.current_page = page;
                    }
                    "navigated" => {
                        let page: String = action.arg().unwrap();
                        state.page_history.push(page);
                    }
                    "other_action" => {
                        state.selection = (1, 1);
                    }
                    _ => {}
                }
                println!("REDUCED {:?}", state);
            },
            vec![NavigationMiddlware::new()],
        );

        select!(|state| state.current_page => |state| {
            println!("select current_page");
            let current_page = &state.current_page;
            unsafe {
                if SELECTOR_CALLS.len() == 0 {
                    assert_eq!(current_page, "home");
                }
                SELECTOR_CALLS.push("select current page");
            }
        });

        select!(|state| state.page_history => |state| {
            let page_history = &state.page_history;
            unsafe {
                if SELECTOR_CALLS.len() == 2 {
                    assert_eq!(page_history, &vec!["settings".to_string()]);
                }
                SELECTOR_CALLS.push("select page history");
            }
        });

        println!("Send dispatches");
        dispatch!("navigate", "settings");
        dispatch!("other_action");

        unsafe {
            assert_eq!(
                SELECTOR_CALLS,
                vec![
                    // selectors are called when defined
                    "select current page",
                    // selectors are called when defined
                    "select page history",
                    // dispatch call
                    "select current page",
                    // middleware call
                    "select page history",
                ]
            );

            assert_eq!(
                REDUCER_CALLS,
                vec![
                    // gstore calls init on itself
                    "init".to_string(),
                    // middleware pre reduce
                    "other_action".to_string(),
                    // first dispatch from above
                    "navigate".to_string(),
                    // middleware post reduce
                    "navigated".to_string(),
                    // the second dispatch
                    "other_action".to_string()
                ]
            );
        }
    }
}
