// SPDX-License-Identifier: GPL-3.0-or-later

/// An Action describes a mutation of the global state.
///
/// It has a name which is used for pattern matching and a payload.
/// Since gstore is for the GTK framework these actions are linked
/// to GTK Actions. GTK Actions use their own 'generic value type'
/// called Variant. Thus gstore Actions have a Variant as argument.
///
/// # Examples
///
/// ```rust,ignore
/// // given we receive an action `action` in a reducer
/// // Note: Action::new is not public
/// let action = Action::new("test".into(), Some(128_i32.to_variant()))
/// // get a rust value from it by using
/// if let Some(number) = action.arg::<i32>() {
///     // do stuff
/// }
/// ```
#[derive(Debug, Clone)]
pub struct Action {
    name: String,
    #[cfg(feature = "latest-gtk")]
    argument: Option<glib::Variant>,
}

impl Action {
    /// Create a new action.
    ///
    /// **NOTE:**
    ///
    /// **This is a placeholder since rustdoc server can not compile latest gtk yet
    /// The actual signature looks like this:**
    /// ```rust
    /// fn new(name: String, argument: Option<glib::Variant>) -> Self
    /// ```
    #[cfg(not(feature = "latest-gtk"))]
    pub(crate) fn new(name: String, argument: Option<String>) -> Self {
        Action { name }
    }

    /// Create a new action.
    #[cfg(feature = "latest-gtk")]
    pub(crate) fn new(name: String, argument: Option<glib::Variant>) -> Self {
        Action { name, argument }
    }

    /// Get the name of the action
    pub fn name(&self) -> &str {
        &self.name
    }

    /// Create a new action.
    ///
    /// **NOTE:**
    ///
    /// **This is a placeholder since rustdoc server can not compile latest gtk yet
    /// The actual signature looks like this:**
    /// ```rust
    /// pub fn argument(&self) -> Option<&glib::Variant>
    /// ```
    #[cfg(not(feature = "latest-gtk"))]
    pub fn argument(&self) -> Option<String> {
        None
    }

    /// Get the argument as glib::Variant.
    #[cfg(feature = "latest-gtk")]
    pub fn argument(&self) -> Option<&glib::Variant> {
        self.argument.as_ref()
    }

    /// Try to get the argument as <T>. If either there is no argument
    /// or the argument is of a different type, the returned Option
    /// will be None.
    ///
    /// **NOTE:**
    ///
    /// **This is a placeholder since rustdoc server can not compile latest gtk yet
    /// The actual signature looks like this:**
    /// ```rust
    /// pub fn arg<T: glib::FromVariant + Sized>(&self) -> Option<T>
    /// ```
    #[cfg(not(feature = "latest-gtk"))]
    pub fn arg<T: Sized>(&self) -> Option<String> {
        None
    }

    /// Try to get the argument as <T>. If either there is no argument
    /// or the argument is of a different type, the returned Option
    /// will be None.
    #[cfg(feature = "latest-gtk")]
    pub fn arg<T: glib::FromVariant + Sized>(&self) -> Option<T> {
        if self.argument().is_some() {
            self.argument().unwrap().get()
        } else {
            panic!("Action '{}' has no argument", self.name());
        }
    }
}

impl std::fmt::Display for Action {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.argument.is_some() {
            let arg = self.argument.as_ref().unwrap();
            let ty = arg.type_().to_str();
            let arg = arg.to_string();
            write!(f, "( {}, {}, {} )", self.name(), ty, arg)
        } else {
            write!(f, "( {} )", self.name())
        }
    }
}

#[cfg(test)]
mod tests {

    use glib::ToVariant;

    use super::*;

    #[test]
    pub fn test_action_display_with_args() {
        let action = Action::new("test".into(), Some(("foo", "bar").to_variant()));
        assert_eq!(action.to_string(), "( test, (ss), ('foo', 'bar') )");

        let action = Action::new("test".into(), Some(1.to_variant()));
        assert_eq!(action.to_string(), "( test, i, 1 )");

        let mut map = std::collections::HashMap::new();
        map.insert("test1", 1.to_variant());
        map.insert("test2", 2.to_variant());
        map.insert("test4", 4.to_variant());
        let action = Action::new("awesome-action".into(), Some(map.to_variant()));
        assert!(action.to_string().starts_with("( awesome-action, a{sv}, "));
        assert!(action.to_string().contains("'test1': <1>"));
        assert!(action.to_string().contains("'test2': <2>"));
        assert!(action.to_string().contains("'test4': <4>"));
    }

    #[test]
    pub fn test_action_display_without_args() {
        let action = Action::new("test".into(), None);
        assert_eq!(action.to_string(), "( test )");
    }
}
