use gstore::{self, dispatch, select, store};

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct State {
    current_page: String,
    page_history: Vec<String>,
    selection: (u32, u32),
}

impl Default for State {
    fn default() -> Self {
        Self {
            current_page: "home".to_string(),
            page_history: vec![],
            selection: Default::default(),
        }
    }
}

mod dispatch_during_reduce {
    use super::*;

    store!(State);

    static mut SELECTOR_CALLS: u8 = 0;

    #[test]
    fn test() {
        env_logger::try_init().ok();

        init_store(
            |action, state| match action.name() {
                "navigate" => {
                    let page: String = action.arg().unwrap();
                    state.current_page = page;
                }
                "navigated" => {
                    let page: String = action.arg().unwrap();
                    state.page_history.push(page);
                }
                "other_action" => {
                    state.selection = (1, 1);
                }
                _ => {}
            },
            Vec::new(),
        );

        select!(|state| state.current_page => |state| {
            let current_page = &state.current_page;
            unsafe {
                if SELECTOR_CALLS == 0 {
                    assert_eq!(current_page, "home");
                } else {
                    assert_eq!(current_page, "settings");
                }
            }
            dispatch!("navigated", current_page);
            unsafe {
                SELECTOR_CALLS += 1;
            }
        });

        select!(|state| state.page_history => |state| {
            let page_history = &state.page_history;
            unsafe {
                if SELECTOR_CALLS == 1 {
                    assert_eq!(page_history, &vec!["home".to_string()]);
                } else {
                    assert_eq!(page_history, &vec!["home".to_string(), "settings".to_string()]);
                }

                SELECTOR_CALLS += 1;
            }
        });

        dispatch!("navigate", "settings");
        dispatch!("other_action");

        unsafe {
            println!("{:?}", STORE);
            assert_eq!(
                SELECTOR_CALLS,
                2 /* initial calls */ + 2 /* from dispatch navigate */
            );
        }
    }
}

mod middleware_dispatches_in_initialization {
    use super::*;

    store!(State);

    static mut MIDDLEWARE_CALLS: Vec<String> = Vec::new();

    struct TestMiddleware {}
    impl TestMiddleware {
        pub fn new() -> Box<Self> {
            Box::new(Self {})
        }
    }
    impl gstore::Middleware<State> for TestMiddleware {
        fn pre_reduce(&self, action: &gstore::Action, _s: &State) {
            unsafe {
                MIDDLEWARE_CALLS.push(action.name().into());
            }
        }
        fn post_reduce(&self, action: &gstore::Action, _s: &State) {
            if action.name() == gstore::INIT {
                dispatch!("foo");
            }
        }
    }

    #[test]
    fn test() {
        env_logger::try_init().ok();

        init_store(
            |action, state| match action.name() {
                "select" => {
                    state.selection = (1, 1);
                }
                _ => {}
            },
            vec![TestMiddleware::new()],
        );

        unsafe {
            assert_eq!(
                MIDDLEWARE_CALLS,
                vec![String::from("init"), String::from("foo")]
            )
        }
    }
}
