use std::sync::Arc;
use std::sync::Mutex;

#[cfg(feature = "latest-gtk")]
use gdk4::prelude::ActionExt;

#[cfg(feature = "widgets")]
use glib::ObjectExt;
#[cfg(feature = "widgets")]
use libadwaita as adw;

use crate::print_perf;

mod action;
pub use action::*;

struct Selector<S: std::fmt::Debug + Clone + Default + 'static> {
    name: String,
    full: bool,
    selector: Box<dyn Fn(&S) -> S + 'static>,
    last_state: S,
    callback: Arc<dyn Fn(S) + 'static>,
}

impl<S: std::fmt::Debug + Clone + Default + 'static> Selector<S> {
    fn new(
        name: &str,
        full: bool,
        selector: impl Fn(&S) -> S + 'static,
        last_state: S,
        callback: impl Fn(S) + 'static,
    ) -> Self {
        Selector {
            name: name.to_string(),
            full,
            selector: Box::new(selector),
            last_state,
            callback: Arc::new(callback),
        }
    }
}

impl<S: std::fmt::Debug + Clone + Default + 'static> std::fmt::Debug for Selector<S> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("Selector")
            .field("name", &self.name)
            .finish()
    }
}

/// A Middleware allows to handle actions before and after state mutation.
///
/// This allows implementing side effects for actions.
pub trait Middleware<S: std::fmt::Debug + Clone + Default + PartialEq + Eq + 'static> {
    fn pre_reduce(&self, _: &Action, _: &S) {}
    fn post_reduce(&self, _: &Action, _: &S) {}
}

/// Sending Actions from GTK widgets via dispatch() performs some checks for
/// prohibiting internal update loops.
#[cfg(feature = "widgets")]
pub trait DispatchWidget: glib::IsA<gtk::Widget> {
    fn dispatch(&self, action: &str, argument: Option<&dyn glib::ToVariant>) {
        crate::check_internal!(self);
        gtk::prelude::WidgetExt::activate_action(
            self,
            &format!("app.{}", action),
            argument.map(|a| a.to_variant()).as_ref(),
        );
    }
}

#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::AppChooser {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::AppChooserButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ApplicationWindow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Box {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Button {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Calendar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::CenterBox {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ColorButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ComboBox {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Dialog {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::DragIcon {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::DrawingArea {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::DropDown {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Editable {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::EmojiChooser {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Entry {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Expander {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Fixed {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::FlowBox {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::FontButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Frame {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Grid {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::GridView {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::HeaderBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::IconView {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Image {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::InfoBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Label {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::LevelBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::LinkButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ListBox {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ListBoxRow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::LockButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::MenuButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::MessageDialog {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Notebook {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::PasswordEntry {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Popover {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::PopoverMenu {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Revealer {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ScrolledWindow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::SearchBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::SearchEntry {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::SpinButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Spinner {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Stack {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::StackSidebar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::StackSwitcher {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Statusbar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Switch {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Text {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::ToggleButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Widget {}
#[cfg(feature = "widgets")]
impl DispatchWidget for gtk::Window {}

#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ActionRow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Carousel {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Clamp {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ComboRow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ExpanderRow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Flap {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::HeaderBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Leaflet {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::PreferencesGroup {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::PreferencesRow {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::SplitButton {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Squeezer {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::Swipeable {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::TabBar {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ViewStack {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ViewSwitcher {}
#[cfg(feature = "widgets")]
impl DispatchWidget for adw::ViewSwitcherBar {}

/// A global Store
///
/// The store holds global app app data with static lifetime.
///
/// It can be manipulated by dispatching actions (See [Action](crate::Action)).
///
/// Each action is handled by the reducer - A function which matches the
/// action and has a mutable ref to the state in the store.
///
/// The reducer must not have any side effects (E.g. saving the state to a file).
///
/// Sideeffects are implemented in middlewares (See [Middleware](crate::Middleware)).
pub struct Store<S: std::fmt::Debug + Clone + Default + PartialEq + Eq + 'static> {
    state: Arc<Mutex<S>>,
    reducer: Arc<Mutex<Box<dyn Fn(&Action, &mut S) + 'static>>>,
    selectors: Arc<Mutex<Vec<Selector<S>>>>,
    middlewares: Arc<Mutex<Arc<Vec<Box<dyn Middleware<S>>>>>>,
}

impl<S: std::fmt::Debug + Clone + Default + PartialEq + Eq + 'static> std::fmt::Debug for Store<S> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("Store")
            .field("state", &self.state)
            .field("selectors", &self.selectors)
            .finish()
    }
}

impl<S: std::fmt::Debug + Clone + Default + PartialEq + Eq> Store<S> {
    /// Create a new Store.
    pub fn new(
        reducer: impl Fn(&Action, &mut S) + 'static,
        middlewares: Vec<Box<dyn Middleware<S>>>,
    ) -> Self {
        let state = Default::default();
        Store {
            state: Arc::new(Mutex::new(state)),
            reducer: Arc::new(Mutex::new(Box::new(reducer))),
            selectors: Arc::new(Mutex::new(Default::default())),
            middlewares: Arc::new(Mutex::new(Arc::new(middlewares))),
        }
    }

    fn select_internal(
        &self,
        name: &str,
        selector: impl Fn(&S) -> S + 'static,
        callback: impl Fn(S) + 'static,
        select_full_state: bool,
    ) {
        let state = self.state.lock().unwrap();
        let s = state.clone();
        drop(state);
        callback(s);

        let mut selectors = self.selectors.lock().unwrap();
        let state = self.state.lock().unwrap();

        let last_selector_state = selector(&state);
        trace!("Adding selector for 'state[{}]'.", name);
        selectors.push(Selector::new(
            name,
            select_full_state,
            selector,
            last_selector_state,
            callback,
        ));
    }

    /// Select a slice of the state where everything else is Default::default().
    pub fn select(
        &self,
        name: &str,
        selector: impl Fn(&S) -> S + 'static,
        callback: impl Fn(S) + 'static,
    ) {
        print_perf!(
            self.select_internal(name, selector, callback, false);
            format!("Call selector {}", name)
        )
    }

    /// Select a slice of the state with receiving the whole state.
    pub fn select_full(
        &self,
        name: &str,
        selector: impl Fn(&S) -> S + 'static,
        callback: impl Fn(S) + 'static,
    ) {
        print_perf!(
            self.select_internal(name, selector, callback, true);
            format!("Call full state selector {}", name)
        )
    }

    /// Dispatch the given arguments as an [Action](crate::Action).
    ///
    /// **NOTE:**
    ///
    /// **This is a placeholder since rustdoc server can not compile latest gtk yet
    /// The actual signature looks like this:**
    /// ```rust
    /// pub fn dispatch(&self, action: String, argument: Option<glib::Variant>)
    /// ```
    #[cfg(not(feature = "glib"))]
    pub fn dispatch(&self, action: String, argument: Option<String>) {}

    /// Dispatch the given arguments as an [Action](crate::Action).
    #[cfg(feature = "glib")]
    pub fn dispatch(&self, action: String, argument: Option<glib::Variant>) {
        let m = format!("Dispatch action {:?} with argument {:?}", action, argument);
        print_perf!(
            {
                let action = Action::new(action, argument);

                trace!("Locking store...");
                let mut state_lock = self.state.lock().unwrap();
                let reducer = self.reducer.lock().unwrap();
                let mut selectors = self.selectors.lock().unwrap();
                let middlewares_lock = self.middlewares.lock().unwrap();
                let middlewares = middlewares_lock.clone();
                drop(middlewares_lock);

                let state = &mut *state_lock;
                trace!(
                    "Reduce action {:?} for state {:?}.",
                    action.name().to_string(),
                    state
                );

                print_perf!(
                    for middleware in middlewares.as_ref() {
                        middleware.pre_reduce(&action, &state);
                    };
                    format!("Call {} middlewares pre_reduce", middlewares.len())
                );

                print_perf!(
                    reducer(&action, state);
                    format!("Reduce action {:?}", action.name())
                );
                trace!("Reduced to: {:?}", state);

                let mut selectors_to_call = Vec::new();
                print_perf!(
                    {
                        for selector in selectors.iter_mut() {
                            let sel = &selector.selector;
                            let selected_state = sel(&state);
                            if selector.last_state != selected_state {
                                selector.last_state = selected_state.clone();
                                let callb = &selector.callback;

                                if selector.full {
                                    selectors_to_call.push((selector.name.clone(), callb.clone(), state.clone()));
                                } else {
                                    selectors_to_call.push((selector.name.clone(), callb.clone(), selected_state));
                                }
                            }
                        }
                    };
                    format!("Determine n of {} selectors for action {:?}", selectors.len(), action.name())
                );

                let state_copy_for_middleware = state_lock.clone();
                drop(state_lock);
                drop(reducer);
                drop(selectors);
                trace!("Freed store.");

                let selectors_to_call_len = selectors_to_call.len();
                print_perf!(
                    for (name, callback, state) in selectors_to_call {
                        trace!("Run selector 'state[{}]'.", &name);
                        callback(state)
                    };
                    format!("Call {} selectors", selectors_to_call_len)
                );

                print_perf!(
                    for middleware in middlewares.as_ref() {
                        middleware.post_reduce(&action, &state_copy_for_middleware);
                    };
                    format!("Call {} middlewares post_reduce", middlewares.len())
                );
            };
            m
        )
    }

    /// Delegate GTK actions as gstore::Actions to this store.
    ///
    /// **NOTE:**
    ///
    /// **This is a placeholder since rustdoc server can not compile latest gtk yet
    /// The actual signature looks like this:**
    /// ```rust
    /// pub fn delegate(&'static self) -> glib::Sender<(gdk4::gio::SimpleAction, Option<glib::Variant>)>
    /// ```
    #[cfg(not(feature = "latest-gtk"))]
    pub fn delegate(&'static self) {}

    /// Delegate GTK actions as gstore::Actions to this store.
    #[cfg(feature = "latest-gtk")]
    pub fn delegate(
        &'static self,
    ) -> glib::Sender<(gdk4::gio::SimpleAction, Option<glib::Variant>)> {
        let (sender, receiver) = glib::MainContext::channel::<(
            gdk4::gio::SimpleAction,
            Option<glib::Variant>,
        )>(glib::PRIORITY_HIGH);
        receiver.attach(None, move |(action, argument)| {
            debug!("Delegate gtk action {:?} {:?}.", action, argument);
            self.dispatch(action.name().to_string(), argument);
            glib::Continue(true)
        });
        sender
    }
}

#[macro_export]
macro_rules! dispatch {
    (
        $action:expr $(, $argument:expr)?
    ) => {{
        let action = $action;
        let mut argument = None;
        $(
        argument = Some(glib::ToVariant::to_variant(&$argument));
        )?
        store().dispatch(action.to_string(), argument);
    }};
}

#[macro_export]
macro_rules! select_state {
    (
        $name:expr, $selector:expr, $callback:expr
    ) => {
        store().select($name, $selector, $callback);
    };
}

#[macro_export]
macro_rules! select {
    (
        |$state:ident| $selector:expr => $callback:expr
    ) => {
        store().select(
            stringify!($selector),
            |s| {
                let mut $state: State = Default::default();
                let sel = |$state: &State| $selector.clone();
                $selector = sel(s);
                $state
            },
            $callback,
        );
    };
}

#[macro_export]
macro_rules! select_typed {
    (
        $state_type:ty |$state:ident| $selector:expr => $callback:expr
    ) => {
        store().select(
            stringify!(state.$selector),
            |s| {
                let mut $state: $state_type = Default::default();
                let sel = |$state: &$state_type| $selector.clone();
                $selector = sel(s);
                $state
            },
            $callback,
        );
    };
}

#[macro_export]
macro_rules! args {
    (
        $($field:ident: $value:expr),*$(,)?
    ) => {
        {
            let mut map = std::collections::HashMap::new();
            $(
            map.insert(stringify!($field), $value.to_variant());
            )*
            map
        }
    };
}

#[macro_export]
macro_rules! store {
    (
        $state:ty
    ) => {
        pub type Store = gstore::Store<$state>;

        static mut STORE: Option<Store> = None;

        /// Initialize the store and global state
        ///
        /// # Arguments
        /// - initializer: Initialize the state.
        /// - reducer: Handle actions and mutate the state
        /// - middlewares: Pre- and post-reduce handlers
        pub fn init_store(
            reducer: impl Fn(&gstore::Action, &mut $state) + 'static,
            middlewares: Vec<Box<dyn gstore::Middleware<$state>>>,
        ) {
            unsafe {
                STORE = Some(Store::new(reducer, middlewares));
                STORE.as_ref().unwrap().dispatch(gstore::INIT.into(), None);
            }
        }

        /// Get a static reference to the store
        pub fn store() -> &'static Store {
            unsafe { STORE.as_ref().expect("Store is not initialized!") }
        }
    };
}

#[macro_export]
macro_rules! set_internal {
    (
        $object:ident => $setter:expr
    ) => {
        unsafe {
            $object.set_data("internal", true);
        }
        $setter;
        unsafe {
            $object.set_data("internal", false);
        }
    };
}

#[macro_export]
macro_rules! check_internal {
    (
        $object:ident
    ) => {
        unsafe {
            if let Some(internal) = $object.data::<bool>("internal") {
                if *internal.as_ref() {
                    return;
                }
            }
        }
    };
}
