/// Creates a GTK box.
///
/// # See Also
/// - [widget Macro](macro.widget.html)
/// - [BoxBuilder in gtk-rs](https://gtk-rs.org/docs/gtk/struct.BoxBuilder.html)
/// - [Box in gtk-rs](https://gtk-rs.org/docs/gtk/struct.Box.html)
///
/// # Examples
///
/// ```rust,no_run
/// # #[macro_use]
/// # extern crate gstore;
/// # use gstore::prelude::*;
/// # use gtk::prelude::*;
/// #
/// # slice! { CountState { count: i64 = 0 } CountAction { Clicked } }
/// # fn reduce_count(action: CountAction, state: CountState) -> CountState { state }
/// # store! { count: Count = crate::{CountState, CountAction, reduce_count} }
/// #
/// # fn main() {
/// #     let store = Store::new(|a,s| s, vec![]);
///
/// gbox! [ horizontal
///     label!(just "Label A")
///     label!(just "Label B")
/// ];
///
/// gbox! [ vertical
///     label!(just "Label A")
///     label!(just "Label B")
/// ];
///
/// gbox! {
///     properties {
///         orientation: gtk::Orientation::Horizontal
///     }
///     children [
///         label!(just "Label A")
///         label!(just "Label A")
///     ]
/// };
/// # }
/// ```
#[macro_export]
macro_rules! gbox {
    (
        vertical $($child:expr)*
    ) => {{
        let gbox = gtk::BoxBuilder::new()
            .visible(true)
            .orientation(gtk::Orientation::Vertical)
            .build();
        widget!(gbox: gtk::Box {
            properties {

            }
            children [
                $($child)*
            ]
        });
        gbox
    }};
    (
        horizontal $($child:expr)*
    ) => {{
        let gbox = gtk::BoxBuilder::new()
            .visible(true)
            .orientation(gtk::Orientation::Horizontal)
            .build();
        widget!(gbox: gtk::Box {
            properties {

            }
            children [
                $($child)*
            ]
        });
        gbox
    }};
        (
        $($text:tt)*
    ) => {{
        let gbox = gtk::BoxBuilder::new().visible(true).build();
        widget!(gbox: gtk::Box { $($text)* });
        gbox
    }}
}
