/// Creates a new ApplicationWindow.
///
/// # See Also
/// - [widget Macro](macro.widget.html)
/// - [StackBuilder in gtk-rs](https://gtk-rs.org/docs/gtk/struct.StackBuilder.html)
/// - [Stack in gtk-rs](https://sgtk-rs.org/docs/gtk/struct.Stack.html)
///
/// # Examples
///
/// ```rust,no_run
/// # #[macro_use]
/// # extern crate gstore;
/// # use gstore::prelude::*;
/// # use gtk::prelude::*;
/// #
/// # slice! { CountState { count: i64 = 0 } CountAction { Clicked } }
/// # fn reduce_count(action: CountAction, state: CountState) -> CountState { state }
/// # store! { count: Count = crate::{CountState, CountAction, reduce_count} }
/// #
/// # fn main() {
/// #     let store = Store::new(|a,s| s, vec![]);
/// application_window! {
///     properties {
///         default_width: 400
///         default_height: 400
///     }
///     connections {
///         (store) connect_destroy: move |_| store.dispatch(Action::Stop)
///     }
///     children [
///         label!(just "Test Label")
///     ]
///     // styles "../styles"
///     title_bar title_bar! {}
/// };
/// # }
/// ```
#[macro_export]
macro_rules! application_window {
    (
        $(properties {
            $($properties:tt)*
        })?
        $(connections {
            $($connections:tt)*
        })?
        $(children [
            $($children:tt)*
        ])?
        $(styles $styles_path:literal)?
        $(title_bar $title_bar:expr)?

    ) => {{
        let window = gtk::ApplicationWindowBuilder::new().build();

        widget!(window: gtk::ApplicationWindow {
            $(properties {
                $($properties)*
            })?
            $(connections {
                $($connections)*
            })?
            $(children [
                $($children)*
            ])?
        });


        $(
        window.set_titlebar(Some(&$title_bar));
        )?

        $(
        let css = include_str!($styles_path);
        let css_provider = gtk::CssProvider::new();
        css_provider
            .load_from_data(css.as_bytes())
            .expect("Could not load css data.");
        let screen = gdk::Screen::get_default().expect("Could not get default screen.");
        gtk::StyleContext::add_provider_for_screen(
            &screen,
            &css_provider,
            gtk::STYLE_PROVIDER_PRIORITY_APPLICATION,
        );
        )?


        window.show();
        window
    }}
}
