//! # gstore
//!
//! gstore is a library that provides global and local state management for GTK Apps as well as a
//! bunch of macros to build GTK widgets.
//!
//! gstore is inspired by react/redux. If you are familiar with these frameworks you might find a
//! lot of their concepts used in gstore.
//!
//! # Usage
//! ```rust,no_run
//! #[macro_use]
//! extern crate gstore;
//! use gtk::prelude::*;
//! use gstore::prelude::*;
//!
//! // slice.rs
//! // -----------------------------------------------------------------
//! // define a slice
//!
//! slice! {
//!     CountState { count: i64 = 0 }
//!     CountAction { Increment, Decrement }
//! }
//!
//! // define a reducer on that slice
//! fn reduce_count(action: CountAction, state: CountState) -> CountState {
//!     match action {
//!         CountAction::Increment => CountState { count: state.count + 1, ..state },
//!         CountAction::Decrement => CountState { count: state.count - 1, ..state }
//!     }
//! }
//!
//! // define a selector for your slice (the state is the Root state)
//! fn select_count(state: &crate::State) -> i64 { state.counting.count }
//!
//! // define functions for convenient action dispatching
//! fn increment() -> Action {
//!     crate::Action::Counting(CountAction::Increment)
//! }
//!
//! fn decrement() -> Action {
//!     crate::Action::Counting(CountAction::Decrement)
//! }
//!
//! // store.rs
//! // -----------------------------------------------------------------
//! // combine slices in your store
//!
//! store! {
//!     counting: Counting = crate::{CountState, CountAction, reduce_count}
//! }
//!
//! fn main() {
//!
//!     let logging_middleware = middleware(|store, next, action: Action| {
//!         println!("Handling action {:?}", action);
//!         next(store, action.clone());
//!         println!("Handled action {:?}", action);
//!     });
//!
//!     let store: Store = Store::new(root_reducer, vec![logging_middleware]);
//!     
//!     gstore::gtk::run(store, |store, _| {
//!         let window = window(store.clone());
//!         store.dispatch(Action::Start);
//!         window
//!     })
//! }
//!
//! // window.rs
//! // -----------------------------------------------------------------
//! // define window component
//!
//! use_state! {
//!     [message: String, set_message] = "The current count:".to_string()
//! }
//!
//! fn window(store: Store) -> gtk::ApplicationWindow {
//!     application_window! {
//!         properties {
//!             default_width: 400
//!             default_height: 400
//!         }
//!         children [
//!             label! {
//!                 properties {
//!                     label: message()
//!                 }
//!             }
//!         ]
//!     }
//! }
//!
//! ```
extern crate log;

pub use once_cell;

pub mod gtk;
pub mod l10n;
pub mod store;
pub mod widgets;

// helpers
pub mod use_state;

pub mod prelude {
    pub use crate::gtk::*;
    pub use crate::l10n::*;
    pub use crate::store::*;
    pub use crate::use_state::*;
    pub use crate::widgets::*;
}
