/// Creates a GTK stack.
///
/// # See Also
/// - [widget Macro](macro.widget.html)
/// - [StackBuilder in gtk-rs](https://gtk-rs.org/docs/gtk/struct.StackBuilder.html)
/// - [Stack in gtk-rs](https://gtk-rs.org/docs/gtk/struct.Stack.html)
///
/// # Examples
///
/// ```rust,no_run
/// # #[macro_use]
/// # extern crate gstore;
/// # use gstore::prelude::*;
/// # use gtk::prelude::*;
/// #
/// # slice! { CountState { count: i64 = 0 } CountAction { Clicked } }
/// # fn reduce_count(action: CountAction, state: CountState) -> CountState { state }
/// # store! { count: Count = crate::{CountState, CountAction, reduce_count} }
/// #
/// # fn main() {
/// #     let store = Store::new(|a,s| s, vec![]);
/// stack! {
///     class "css-class"
///     properties {
///         visible_child_name: "test"
///     }
///     connections {
///         connect_property_visible_child_notify: move |s| {
///             println!("Changed child: {:?}", s.get_visible_child_name().unwrap())
///         }
///     }
///     children [
///         label!(just "Test Label")
///     ]
///     children_properties [
///         // (Option<GTK widget name>, Option<Stack item title>, Option<icon name>)
///         (Some("child1"), Some("Child One".to_string()), Some("web-browser-symbolic"))
///     ]
/// };
/// # }
/// ```
#[macro_export]
macro_rules! stack {
    (
        $(class $class:literal)?

        $(properties $($stateful:ident)? {
            $( $prop:ident: $value:expr )*
        })?

        $(connections {
            $(
                $( ($store:ident) => )? $connector:ident: $connection:expr
            )*
        })?

        $(children [
            $($child:expr)*
        ])?

        $(children_properties [
            $( ($child_name:expr, $child_title:expr, $child_icon:expr) )*
        ])?
    ) => {{
        let widget = gtk::StackBuilder::new().visible(true).build();

        widget!(widget: gtk::Stack {
            $(class $class)?
            $(connections {
                $(
                    $(($store) =>)? $connector: $connection
                )*
            })?
            $(children [
                $($child)*
            ])?
        });

        $(
        let mut children_props: Vec<(Option<&str>, Option<String>, Option<&str>)> = Vec::new();
        $( children_props.push(($child_name, $child_title, $child_icon));
        )*

        if children_props.len() != widget.get_children().len() {
            panic!("Stack has {} children but {} child property definitions.", widget.get_children().len(), children_props.len());
        }

        for (i, child) in widget.get_children().iter().enumerate() {
            let props = children_props.get(i).unwrap();
            widget.set_child_name(child, props.0.as_deref());
            widget.set_child_title(child, props.1.as_deref());
            widget.set_child_icon_name(child, props.2.as_deref());
        }
        )?

        widget!(widget: gtk::Stack {
            $(properties $($stateful)? {
                $( $prop: $value )*
            })?
        });



        widget
    }};
    (
        $($child:expr)*
    ) => {{
        let widget = gtk::StackBuilder::new().visible(true).build();
        widget! (widget: gtk::Stack {
            children [
                $($child)*
            ]
        });
        for (i, child) in widget.get_children().iter().enumerate() {
            widget.set_child_name(child, format!("{}.{}", child.get_widget_name().as_str(), i).as_str().into());
            widget.set_child_title(child, format!("{}.{}", child.get_widget_name().as_str(), i).as_str().into());
        }
        widget
    }}
}
