/// gstore makro for the GTK label.
///
/// ## See Also
/// - [widget Macro](macro.widget.html)
/// - [LabelBuilder in gtk-rs](https://gtk-rs.org/docs/gtk/struct.LabelBuilder.html)
/// - [Label in gtk-rs](https://gtk-rs.org/docs/gtk/struct.Label.html)
///
/// ## Examples
///
/// Long example:
/// ```rust,no_run
/// # #[macro_use]
/// # extern crate gstore;
/// # use gstore::prelude::*;
/// # use gtk::prelude::*;
/// #
/// # slice! { CountState { count: i64 = 0 } CountAction { Clicked } }
/// # fn reduce_count(action: CountAction, state: CountState) -> CountState { state }
/// # store! { count: Count = crate::{CountState, CountAction, reduce_count} }
/// #
/// # use_state! {
/// #     [count: u32, set_count] = 0
/// # }
/// # fn main() {
/// #    let store = Store::new(|a,s| s, vec![]);
/// label! {
///     // set css label
///     class "my-label-css-class"
///     // define gtk label properties
///     properties stateful {
///         label: format!("{}", count())
///     }
/// };
/// # }
/// ```
///
/// Short example without local state:
/// ```rust,no_run
/// # #[macro_use]
/// # extern crate gstore;
/// # use gstore::prelude::*;
/// # use gtk::prelude::*;
/// # fn main() {
/// label!(just "Test");
/// # }
/// ```
#[macro_export]
macro_rules! label {
    (
        just $text:expr
    ) => {{
        gtk::LabelBuilder::new()
            .visible(true)
            .label($text)
            .build()
    }};

    (
        select $state:ident -> {$getter:expr}
        $(css $class:expr)?
    ) => {{
        let label = gtk::LabelBuilder::new().visible(true).build();
        widget!(label: gtk::Label {
            with $state
            $(class $class)?
            properties {
                label: $getter
            }
        });
        label
    }};

    (
        $($text:tt)*
    ) => {{
        let label = gtk::LabelBuilder::new().visible(true).build();
        widget!(label: gtk::Label { $($text)* });
        label
    }}
}
