/// GTK application use gettext for internationalization.
/// This module provides glue code to initialize a text domain using gettext.
use std::io::BufRead;

/// Initialized a text domain for the given name.
///
/// Please checkout gettext https://www.gnu.org/software/gettext/ to understand how it works.
///
/// To have a proper setup of the required .po files and an integration into your crates
/// build process check the build() function in this module.
///
/// Run your app via `LANGUAGE="de_DE:de" LANG="de_DE.utf8" TEXT_DOMAIN="target" cargo run` to test the generated
/// text domain in your target directory and a specific locale.
///
/// # See Also
/// gettext-rs: https://docs.rs/gettext-rs/
///
/// # Arguments
/// domain: The text domain name
/// callback: A callback which provides the loaded locale (depending on the system locale)
///
pub fn init(domain: &str, callback: impl Fn(String)) {
    let textdomain = match std::env::var("TEXT_DOMAIN") {
        Ok(path) => gettextrs::TextDomain::new(domain)
            .skip_system_data_paths()
            .push(&path)
            .init(),
        Err(_) => gettextrs::TextDomain::new(domain).init(),
    };
    match textdomain {
        Ok(locale) => match locale {
            Some(locale) => callback(locale),
            None => eprintln!("Warning: No locale was set! Probably /usr/share/locale/*/LC_MESSAGES does not contain a .mo file.")
        },
        Err(e) => match e {
            gettextrs::TextDomainError::InvalidLocale(locale) => eprintln!("Warning: Invalid locale {:?}", locale),
            gettextrs::TextDomainError::TranslationNotFound(locale) => match locale.as_str() {
                "en" => callback("en_US.UTF-8".to_string()),
                _ => eprintln!("Warning: Could not find messages for locale {:?}", locale)
            },
        }
    };
}

/// This build script:
///
/// 1. triggers the required gettext calls to analyze a projects code for used keys and
/// 2. initializes/updates your po files and
/// 3. builds a resulting .mo binary file with your translations
///
/// The po files are located in [project_root]/po.
/// The resulting mo file is located in [project_root]/target/locale/[locale]/LC_MESSAGES
///
/// # Requirements:
/// - This build script requires xgettext to be installed on your system
/// - gstore must be listed as build dependency
/// - The file [project_root]/po/LINGUAS need to exist and list all languages (e.g. de, fr, es) that
/// will be translated to.
/// - The file [project_root]/po/POTFILES.in need to exist and contain a alphanumerical list of the code
/// files to analyze for translation calls.
/// - Translation .po files in [project_root]/po/[locale].po for all languages stated in LINGUAS. If
/// these files do not exist the initial build will create template files in target/po which can be
/// used to start with.
///
/// # Arguments
/// domain: The text domain name which is used as file name for the generated .mo file.
pub fn build_script(domain: &str) {
    println!("cargo:rerun-if-changed=src");

    let potfiles = format!("po/POTFILES.in");
    let target_pot_dir = format!("target/po");

    if let Err(e) = std::fs::create_dir_all(target_pot_dir) {
        println!("cargo:warning={:?}", e);
    }

    for line in read_lines("po/LINGUAS").unwrap() {
        if let Ok(line) = line {
            if !line.starts_with("#") {
                let locale = line;

                let po_file = format!("po/{}.po", locale);
                let pot_file = format!("target/po/{}.pot", locale);

                let target_mo_dir = format!("target/locale/{}/LC_MESSAGES", locale);
                let target_mo = format!("target/locale/{}/LC_MESSAGES/{}.mo", locale, domain);

                if let Err(e) = std::fs::create_dir_all(target_mo_dir) {
                    println!("cargo:warning={:?}", e);
                }

                if let Err(e) = std::process::Command::new("xgettext")
                    .arg("-f")
                    .arg(&potfiles)
                    .arg("-o")
                    .arg(&pot_file)
                    .output()
                {
                    println!("cargo:warning={:?}", e);
                }
                if let Err(e) = std::process::Command::new("msgmerge")
                    .arg(&po_file)
                    .arg(&pot_file)
                    .arg("-U")
                    .output()
                {
                    println!("cargo:warning={:?}", e);
                }
                if let Err(e) = std::process::Command::new("msgfmt")
                    .arg("-o")
                    .arg(&target_mo)
                    .arg(&po_file)
                    .output()
                {
                    println!("cargo:warning={:?}", e);
                }
            }
        }
    }
}

fn read_lines<P>(filename: P) -> std::io::Result<std::io::Lines<std::io::BufReader<std::fs::File>>>
where
    P: AsRef<std::path::Path>,
{
    let file = std::fs::File::open(filename)?;
    Ok(std::io::BufReader::new(file).lines())
}

#[macro_export]
macro_rules! t {
    ($s:expr, $($arg:expr),*) => {{
        let mut s = $s;
        {
            let mut i = 0;
            $(
                s = s.replace(&format!("{{{}}}", i), &format!("{}", $arg));
                i += 1;
            )*
        }
        s
    }};
}
