use gio::prelude::ApplicationExtManual;
use gio::ApplicationExt;

use crate::prelude::LifecycleState;
use std::sync::mpsc::TryRecvError;

/// Runs a gtk main loop while listening to the given store.
pub fn run<F, A, S>(store: crate::store::Store<A, S>, window: F)
where
    F: Fn(crate::store::Store<A, S>) -> gtk::ApplicationWindow + 'static,
    A: Send + Clone + Eq + 'static,
    S: LifecycleState + Clone + Eq + Default + 'static,
{
    if gtk::init().is_err() {
        eprintln!("Failed to initialize GTK.");
        return;
    }

    #[cfg(feature = "with_libhandy")]
    libhandy::init();

    store.register(
        |s1, s2| s1.running() != s2.running(),
        |s| {
            if !s.running() {
                gtk::main_quit()
            }
        },
    );

    {
        let store = store.clone();
        glib::timeout_add_local(100, move || {
            glib::Continue(match store.try_receive() {
                Ok(a) => {
                    store.dispatch(a);
                    store.select(|s| s.running())
                }
                Err(TryRecvError::Empty) => true,
                Err(TryRecvError::Disconnected) => false,
            })
        });
    }

    let gtk_app = gtk::ApplicationBuilder::new().build();

    gtk_app.connect_startup(move |_| {
        window(store.clone());
    });

    gtk_app.connect_activate(move |_| {
        gtk::main();
    });

    gtk_app.run(std::env::args().collect::<Vec<String>>().as_slice());
}
