#[macro_use]
extern crate clap;
extern crate gsof_protocol;

use clap::{App, Arg};
use gsof_protocol::client::machine;
use tracing::info;
use tracing::{span, Instrument, Level};
use tracing_subscriber;

mod validators;
use validators::isnumeric;

#[doc = r#"
This command allows you to connect to a TCP Socket server (ip and port provided)
Can receive any set of data separated by a sign like /*end*/.
So you have to detect or define what is the separator to manage correctly the
stream of data.
With the command /--help/ you can access to the options and default values.
```
datagen-client --help
```
"#]
#[tokio::main]
pub async fn main() {
    tracing_subscriber::fmt::init();
    let span_main = span!(Level::TRACE, "main");
    info!("Starting client for datagen server");
    // obtener parámetros de la línea de comandos

    let matches = App::new("Datagen Network Client CLI")
        .author(crate_authors!())
        .version(crate_version!())
        .about("This is the datagen server cli app")
        .arg(
            Arg::with_name("host")
                .short("h")
                .long("host")
                .takes_value(true)
                .value_name("HOST")
                .default_value("0.0.0.0")
                .case_insensitive(true)
                .help("Give a hostname, url or ip"),
        )
        .arg(
            Arg::with_name("port")
                .short("p")
                .long("port")
                .takes_value(true)
                .value_name("NAME")
                .default_value("8888")
                .validator(isnumeric)
                .help("Give a name for protocol"),
        )
        .get_matches();


    if let Some(host) = matches.value_of("host") {
        info!("host {}", host);
    }
    let host = matches.value_of("host").unwrap().to_string();

    if let Some(port) = matches.value_of("port") {
        info!("port {}", port);
    }
    let port = matches
        .value_of("port")
        .unwrap()
        .to_string()
        .parse::<u16>()
        .unwrap();

    machine::client(&host, &port)
        .instrument(span_main)
        .await;
}
