//
// Copyright 2018 Tamas Blummer
//
//Licensed under the Apache License, Version 2.0 (the "License");
//you may not use this file except in compliance with the License.
//You may obtain a copy of the License at
//
//http://www.apache.org/licenses/LICENSE-2.0
//
//Unless required by applicable law or agreed to in writing, software
//distributed under the License is distributed on an "AS IS" BASIS,
//WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//See the License for the specific language governing permissions and
//limitations under the License.
//
extern crate libc;

use libc::{c_int,c_uchar, c_uint};

#[allow(non_camel_case_types)]
#[derive(PartialEq, Eq, PartialOrd, Ord, Hash, Debug, Clone, Copy)]
#[repr(C)]
pub enum Error {
    ERR_SCRIPT = 0,
    #[allow(dead_code)]
    ERR_TX_INDEX,
    #[allow(dead_code)]
    ERR_TX_SIZE_MISMATCH,
    #[allow(dead_code)]
    ERR_TX_DESERIALIZE,
    #[allow(dead_code)]
    ERR_AMOUNT_REQUIRED,
    #[allow(dead_code)]
    ERR_INVALID_FLAGS
}

#[allow(dead_code)]
pub const VERIFY_NONE : c_uint = 0;
// evaluate P2SH (BIP16) subscripts
pub const VERIFY_P2SH : c_uint = 1 << 0;
// enforce strict DER (BIP66) compliance
pub const VERIFY_DERSIG : c_uint = 1 << 2;
// enforce NULLDUMMY (BIP147)
pub const VERIFY_NULLDUMMY : c_uint = 1 << 4;
// enable CHECKLOCKTIMEVERIFY (BIP65)
pub const VERIFY_CHECKLOCKTIMEVERIFY : c_uint = 1 << 9;
// enable CHECKSEQUENCEVERIFY (BIP112)
pub const VERIFY_CHECKSEQUENCEVERIFY : c_uint = 1 << 10;
// enable WITNESS (BIP141)
pub const VERIFY_WITNESS : c_uint = 1 << 11;

pub const VERIFY_ALL : c_uint = VERIFY_P2SH | VERIFY_DERSIG | VERIFY_NULLDUMMY |
    VERIFY_CHECKLOCKTIMEVERIFY | VERIFY_CHECKSEQUENCEVERIFY | VERIFY_WITNESS;

extern "C" {
    pub fn bitcoinconsensus_version() -> c_int;

    pub fn bitcoinconsensus_verify_script_with_amount(
        script_pubkey:  *const c_uchar,
        script_pubkeylen: c_uint,
        amount: u64,
        tx_to: *const c_uchar,
        tx_tolen: c_uint,
        n_in: c_uint,
        flags: c_uint,
        err: *mut Error) -> c_int;
}

/// Compute flags for soft fork activation heights on the groestlcoin network
pub fn height_to_flags(height: u32) -> u32 {

    let mut flag = VERIFY_NONE;
    if height > 1439422 {
        flag |= VERIFY_P2SH;
    }
    if height > 79999 {
        flag |= VERIFY_DERSIG;
    }
    if height > 2464000 {
        flag |= VERIFY_CHECKLOCKTIMEVERIFY;
    }
    if height > 1439424 {
        flag |= VERIFY_CHECKSEQUENCEVERIFY;
    }
    if height > 1439424 {
        flag |= VERIFY_NULLDUMMY | VERIFY_WITNESS
    }
    flag as u32
}

/// Return libgroestlcoinconsenus version
pub fn version () -> u32 {
    unsafe { bitcoinconsensus_version() as u32 }
}

/// Verify a single spend (input) of a Groestlcoin transaction.
/// # Arguments
///  * spend_output_script: a Groestlcoin transaction output script to be spent, serialized in Groestlcoin's on wire format
///  * amount: The spent output amount in gros
///  * spending_transaction: spending Groestlcoin transaction, serialized in Groestlcoin's on wire format
///  * input_index: index of the input within spending_transaction
/// # Returns
/// OK or Err. Note that amount will only be checked for Segwit transactions.
///
/// # Example
///
/// The (randomly choosen) Groestlcoin transaction [aca326a724eda9a461c10a876534ecd5ae7b27f10f26c3862fb996f80ea2d45d](https://blockchain.info/tx/aca326a724eda9a461c10a876534ecd5ae7b27f10f26c3862fb996f80ea2d45d)
/// spends one input, that is the first output of [95da344585fcf2e5f7d6cbf2c3df2dcce84f9196f7a7bb901a43275cd6eb7c3f](https://blockchain.info/tx/95da344585fcf2e5f7d6cbf2c3df2dcce84f9196f7a7bb901a43275cd6eb7c3f) with a value of 630482530 satoshis
///
/// The spending transaction in wire format is:
///
/// `
/// spending = 02000000013f7cebd65c27431a90bba7f796914fe8cc2ddfc3f2cbd6f7e5f2fc854534da95000000006b483045022100de1ac3bcdfb0332207c4a91f3832bd2c2915840165f876ab47c5f8996b971c3602201c6c053d750fadde599e6f5c4e1963df0f01fc0d97815e8157e3d59fe09ca30d012103699b464d1d8bc9e47d4fb1cdaa89a1c5783d68363c4dbc4b524ed3d857148617feffffff02836d3c01000000001976a914fc25d6d5c94003bf5b0c7b640a248e2c637fcfb088ac7ada8202000000001976a914fbed3d9b11183209a57999d54d59f67c019e756c88ac6acb0700
/// `
///
/// The script of the first output of the spent transaction is:
///
/// `
/// spent = 76a9144bfbaf6afb76cc5771bc6404810d1cc041a6933988ac
/// `
///
/// The (pseudo code) call:
///
/// `
/// verify(spent, 630482530, spending, 0)
/// `
/// should return OK(())
///
/// **Note** that spent amount will only be checked for Segwit transactions. Above example is not segwit, therefore verify will succeed with any amount.

///
pub fn verify (spent_output: &[u8], amount: u64, spending_transaction: &[u8], input_index: usize) -> Result<(), Error> {
    verify_with_flags (spent_output, amount, spending_transaction, input_index, VERIFY_ALL)
}

/// Same as verify but with flags that turn past soft fork features on or off
pub fn verify_with_flags (spent_output_script: &[u8], amount: u64, spending_transaction: &[u8], input_index: usize, flags: u32) -> Result<(), Error> {
    unsafe {
        let mut error = Error::ERR_SCRIPT;

        let ret = bitcoinconsensus_verify_script_with_amount(
            spent_output_script.as_ptr(),
            spent_output_script.len() as c_uint,
            amount as u64,
            spending_transaction.as_ptr(),
            spending_transaction.len() as c_uint,
            input_index as c_uint,
            flags as c_uint,
            &mut error
        );
        if ret != 1 {
            Err(error)
        } else {
            Ok(())
        }
    }
}

#[cfg(test)]
mod tests {
    extern crate rustc_serialize as serialize;

    use super::*;
    use self::serialize::hex::FromHex;

    #[test]
    fn bitcoinconsensus_test() {
        // a random old-style transaction from the blockchain
        verify_test (
            "76a91457302b581f7cda7c493a82caff1db6062adeee1f88ac",
            "0100000003328850b762ac5e2f468773a07534191b9c1fd8d319fe20b85ed69995ccf4cdb9000000006a47304402204eac084e4f1cabe5760d9f6b8fa87eb95e7ee2ccda2d220742ebaac5ceaaabe8022023c42a7fea4bd950b9ec3b871007fbe2595766ebb78dbc8660c44d164ea6b19e012103126ee3f2d836b170e4758d445050d9093005a4fcbaf9ba486eb83c4e750fea3affffffff023745c4c4abcf1ff556053934e2f89bc7ad72959d9ef8a7b78d8ef7cc134d3c000000006b483045022100878e7da64bcde4e994b1eda4ca07856081939a112da054c9a60f3e5ebdc37df002204abd5014ec0f6d998d8425ba2a253f6bf3f038269a998dca886eea7120258563012103126ee3f2d836b170e4758d445050d9093005a4fcbaf9ba486eb83c4e750fea3affffffff416a94a12c928e17307f4838445ec49be54302bd50dbffb3dd4ea5f9d87e49f0010000006c493046022100dffb1b6feec7b87117b0dc72cedeac210f6b7bd9dcbe79d9726727d5a7407cfb022100a8085b18a9e1bf578c3c2685bada6bf57a00505e4e27e5d6376457b62fbed2f0012103f5b187b0afe7f0a926e2c391bc46d4bbcf65eb2f5796d736ba01362c74327200ffffffff05a991f2a3010000001976a914f9738c2ba9ccde63afa057507063c2f645b1784788ac5e8f8d1e000000001976a914ee0b02f356670588045fd117e472e343531443dd88ac812a460e000000001976a91417e856d911bf5390f3f0e869c0815715f42360ca88aca2253ea90b0000001976a9146be915c43873f10dfb84d229f374907881f0c99c88ac82b07e67000000001976a91438100f9db9cc791747de0f6e2ee58dce7c312a8988ac00000000",
            29337343871, 0
        ).unwrap();

        // a random segwit transaction from the blockchain using P2SH
        verify_test (
            "a9147ef7378df4bd75487d8667b45beedfb8b00ad4eb87",
            "01000000000101451cbe1c7eef6e7964b2848a47d85cf443b42bd1d8b9afc1f6766a29306e02610000000017160014082f5e0047363ab82987aef0efcba1cb37ede914ffffffff0192258a810300000017a914f907c456f4147f2359e5663c50e458fb32e4f78b870247304402206d706aba6cc88c5886ddf27915e760fccf92684fc8cb121e2bcd442b3722dbb1022046220b828c342209b0a19eb25494578b699d3db1b7e6a128067d25a3058ef9380121028b0cdcbe510201328db9b722c4fc663ee586fcbb9229a933c487222e2c9805f200000000",
            15058219059, 0
        ).unwrap();

        // a random segwit transaction from the blockchain using native segwit
        verify_test(
            "0014f2075f97aaef79587e621275f0ba25e47e750e1a",
            "020000000001019e0e6d901a29f49fc217edb0e18c036727115834fb6ada9c764e9740a16a83b90000000000ffffffff0198e6180207000000160014bb380c38f25920fc9946f6a84b6442eebbb77248024730440220786b3e70502d1c0ec300d3144d74b4cbc87fb1168e5821b78404962aaab05e9902203e27204e4ad621802c4191fee6eed8d1b97c7b2ba7e083fb7e4d09380cb7d0a401210223b5e6c2231dc61a800fde397f64d271dc8e65d8fd01a7932d3e4270dd32774e00000000",
            30099980000 , 0
        ).unwrap();

        // a random old-style transaction from the blockchain - WITH WRONG SIGNATURE for the address
        assert!(verify_test (
            "76a9144bfbaf6afb76cc5771bc6404810d1cc041a6933988ff",
            "0100000003328850b762ac5e2f468773a07534191b9c1fd8d319fe20b85ed69995ccf4cdb9000000006a47304402204eac084e4f1cabe5760d9f6b8fa87eb95e7ee2ccda2d220742ebaac5ceaaabe8022023c42a7fea4bd950b9ec3b871007fbe2595766ebb78dbc8660c44d164ea6b19e012103126ee3f2d836b170e4758d445050d9093005a4fcbaf9ba486eb83c4e750fea3affffffff023745c4c4abcf1ff556053934e2f89bc7ad72959d9ef8a7b78d8ef7cc134d3c000000006b483045022100878e7da64bcde4e994b1eda4ca07856081939a112da054c9a60f3e5ebdc37df002204abd5014ec0f6d998d8425ba2a253f6bf3f038269a998dca886eea7120258563012103126ee3f2d836b170e4758d445050d9093005a4fcbaf9ba486eb83c4e750fea3affffffff416a94a12c928e17307f4838445ec49be54302bd50dbffb3dd4ea5f9d87e49f0010000006c493046022100dffb1b6feec7b87117b0dc72cedeac210f6b7bd9dcbe79d9726727d5a7407cfb022100a8085b18a9e1bf578c3c2685bada6bf57a00505e4e27e5d6376457b62fbed2f0012103f5b187b0afe7f0a926e2c391bc46d4bbcf65eb2f5796d736ba01362c74327200ffffffff05a991f2a3010000001976a914f9738c2ba9ccde63afa057507063c2f645b1784788ac5e8f8d1e000000001976a914ee0b02f356670588045fd117e472e343531443dd88ac812a460e000000001976a91417e856d911bf5390f3f0e869c0815715f42360ca88aca2253ea90b0000001976a9146be915c43873f10dfb84d229f374907881f0c99c88ac82b07e67000000001976a91438100f9db9cc791747de0f6e2ee58dce7c312a8988ac00000000",
            0, 0
        ).is_err());

        // a random segwit transaction from the blockchain using P2SH - WITH WRONG AMOUNT
        assert!(verify_test (
            "a9147ef7378df4bd75487d8667b45beedfb8b00ad4eb87",
            "01000000000101451cbe1c7eef6e7964b2848a47d85cf443b42bd1d8b9afc1f6766a29306e02610000000017160014082f5e0047363ab82987aef0efcba1cb37ede914ffffffff0192258a810300000017a914f907c456f4147f2359e5663c50e458fb32e4f78b870247304402206d706aba6cc88c5886ddf27915e760fccf92684fc8cb121e2bcd442b3722dbb1022046220b828c342209b0a19eb25494578b699d3db1b7e6a128067d25a3058ef9380121028b0cdcbe510201328db9b722c4fc663ee586fcbb9229a933c487222e2c9805f200000000",
            900000, 0).is_err());

        // a random segwit transaction from the blockchain using native segwit - WITH WRONG SEGWIT
        assert!(verify_test(
            "0020701a8d401c84fb13e6baf169d59684e17abd9fa216c8cc5b9fc63d622ff8c58f",
            "020000000001019e0e6d901a29f49fc217edb0e18c036727115834fb6ada9c764e9740a16a83b90000000000ffffffff0198e6180207000000160014bb380c38f25920fc9946f6a84b6442eebbb77248024730440220786b3e70502d1c0ec300d3144d74b4cbc87fb1168e5821b78404962aaab05e9902203e27204e4ad621802c4191fee6eed8d1b97c7b2ba7e083fb7e4d09380cb7d0a401210223b5e6c2231dc61a800fde397f64d271dc8e65d8fd01a7932d3e4270dd32774e00000000",
            18393430 , 0
        ).is_err());

    }

    fn verify_test (spent : &str, spending :&str, amount :u64, input: usize) -> Result<(),Error> {
        verify (spent.from_hex().unwrap().as_slice(), amount, spending.from_hex().unwrap().as_slice(), input)
    }

    #[test]
    fn invalid_flags_test() {
        verify_with_flags(&[], 0, &[], 0, VERIFY_ALL + 1).unwrap_err();
    }
}
