/*
usefull docs:
- https://github.com/FreshRSS/FreshRSS/blob/master/p/api/greader.php
- https://github.com/theoldreader/api
- https://feedhq.readthedocs.io/en/latest/api/index.html
- https://www.inoreader.com/developers/
*/

pub mod error;
pub mod models;
#[cfg(test)]
mod tests;

use std::collections::HashMap;

pub use crate::error::{ApiError, ApiErrorKind};
use crate::models::GReaderError;
pub use crate::models::{AuthData, GoogleAuth, InoreaderAuth};
use crate::models::{Feeds, ItemRefs, QuickFeed, Stream, StreamType, Taggings, Unread, User};

use chrono::{Duration, Utc};
use failure::ResultExt;
use log::error;
use models::{AuthInput, OAuthResponse};
use reqwest::header::{HeaderMap, HeaderValue, AUTHORIZATION};
use reqwest::{Client, StatusCode};
use std::sync::Arc;
use std::sync::Mutex;
use url::Url;

pub struct GReaderApi {
    base_uri: Url,
    auth_input: AuthInput,
    auth: Arc<Mutex<AuthData>>,
}

impl GReaderApi {
    /// Create a new instance of the GReaderApi
    pub fn new(url: &Url, auth: AuthData) -> Self {
        GReaderApi {
            base_uri: url.clone(),
            auth_input: AuthInput::Uninitialized,
            auth: Arc::new(Mutex::new(auth)),
        }
    }

    fn get_auth_headers(&self) -> Result<HeaderMap, ApiError> {
        let mut headers = HeaderMap::new();

        match &*self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)?
        {
            AuthData::Uninitialized => return Err(ApiErrorKind::Unknown.into()),
            AuthData::Google(auth_data) => {
                headers.insert(
                    AUTHORIZATION,
                    HeaderValue::from_str(&format!(
                        "GoogleLogin auth={}",
                        &auth_data.auth_token.clone().unwrap()
                    ))
                    .unwrap(),
                );
            }
            AuthData::Inoreader(auth_data) => {
                // check if access_token is still valid
                let expires_in = auth_data.expires_at.signed_duration_since(Utc::now());
                let expired = expires_in.num_seconds() <= 60;

                if expired {
                    return Err(ApiErrorKind::TokenExpired.into());
                }

                headers.insert(
                    AUTHORIZATION,
                    HeaderValue::from_str(&format!("Bearer {}", auth_data.access_token)).unwrap(),
                );
                headers.insert(
                    "AppId",
                    HeaderValue::from_str(&auth_data.client_id).unwrap(),
                );
                headers.insert(
                    "AppKey",
                    HeaderValue::from_str(&auth_data.client_secret).unwrap(),
                );
            }
        };

        Ok(headers)
    }

    async fn get_request(
        &self,
        query: String,
        params: &mut Vec<(String, String)>,
        client: &Client,
    ) -> Result<String, ApiError> {
        let api_url: Url = self.base_uri.join(&query).context(ApiErrorKind::Url)?;

        params.push(("output".to_string(), "json".to_string()));

        let response = client
            .get(api_url.clone())
            .headers(self.get_auth_headers()?)
            .query(&params)
            .send()
            .await
            .context(ApiErrorKind::Http)?;

        let status = response.status();
        let response = response.text().await.context(ApiErrorKind::Http)?;
        if status != StatusCode::OK {
            let error: GReaderError =
                serde_json::from_str(&response).context(ApiErrorKind::Json)?;
            error!("GReader API: {}", error.errors.join("; "));
            return Err(ApiError::parse_error(error));
        }
        Ok(response)
    }

    async fn post_request(
        &self,
        query: String,
        params: &mut Vec<(String, String)>,
        form_params: Option<Vec<(String, String)>>,
        client: &Client,
    ) -> Result<String, ApiError> {
        let api_url: Url = self.base_uri.join(&query).context(ApiErrorKind::Url)?;

        params.push(("output".to_string(), "json".to_string()));

        let response = client
            .post(api_url.clone())
            .headers(self.get_auth_headers()?)
            .query(&params)
            .form(&form_params)
            .send()
            .await
            .context(ApiErrorKind::Http)?;

        let status = response.status();
        let response = response.text().await.context(ApiErrorKind::Http)?;
        if status != StatusCode::OK {
            let error: GReaderError =
                serde_json::from_str(&response).context(ApiErrorKind::Json)?;
            error!("GReader API: {}", error.errors.join("; "));
            return Err(ApiErrorKind::GReader(error).into());
        }
        Ok(response)
    }

    fn chech_ok_response(response: &str) -> Result<(), ApiError> {
        if response == "OK" {
            Ok(())
        } else {
            let error: GReaderError = GReaderError {
                errors: vec![response.to_string()],
            };
            Err(ApiErrorKind::GReader(error).into())
        }
    }

    pub async fn login(
        &mut self,
        auth_input: AuthInput,
        client: &Client,
    ) -> Result<AuthData, ApiError> {
        *self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)? = match &auth_input {
            AuthInput::Uninitialized => return Err(ApiErrorKind::Input.into()),
            AuthInput::Inoreader(input) => {
                let mut map: HashMap<String, String> = HashMap::new();
                map.insert("code".into(), input.auth_code.clone());
                map.insert("redirect_uri".into(), input.redirect_url.clone());
                map.insert("client_id".into(), input.client_id.clone());
                map.insert("client_secret".into(), input.client_secret.clone());
                map.insert("scope".into(), "".into());
                map.insert("grant_type".into(), "authorization_code".into());

                let response = client
                    .post("https://www.inoreader.com/oauth2/token")
                    .form(&map)
                    .send()
                    .await
                    .context(ApiErrorKind::Http)?
                    .text()
                    .await
                    .context(ApiErrorKind::Http)?;

                let oauth_response =
                    OAuthResponse::from_json(&response).context(ApiErrorKind::Json)?;

                let now = Utc::now();
                let token_expires = now + Duration::seconds(oauth_response.expires_in);

                AuthData::Inoreader(InoreaderAuth {
                    client_id: input.client_id.clone(),
                    client_secret: input.client_secret.clone(),
                    access_token: oauth_response.access_token,
                    refresh_token: oauth_response.refresh_token,
                    expires_at: token_expires,
                })
            }
            AuthInput::Google(input) => {
                let path = format!(
                    "accounts/ClientLogin?Email={}&Passwd={}",
                    input.username, input.password
                );

                let api_url: Url = self.base_uri.join(&path).context(ApiErrorKind::Url)?;
                let response = client
                    .post(api_url.clone())
                    .send()
                    .await
                    .context(ApiErrorKind::Http)?;

                let status = response.status();
                let response = response.text().await.context(ApiErrorKind::Http)?;
                if status != StatusCode::OK {
                    return Err(ApiErrorKind::AccessDenied.into());
                }

                // maybe use ini parser
                // currently only accepting the 'Auth' value
                let auth_string = response.lines().nth(2).unwrap();
                let auth_token = auth_string.split('=').nth(1).unwrap();

                let google_get_auth = GoogleAuth {
                    username: input.username.clone(),
                    password: input.password.clone(),
                    auth_token: Some(auth_token.to_string()),
                    post_token: None,
                };
                *self
                    .auth
                    .lock()
                    .map_err(|_| ApiErrorKind::InternalMutabilty)? =
                    AuthData::Google(google_get_auth);

                let mut response = self
                    .get_request("reader/api/0/token".to_string(), &mut Vec::new(), client)
                    .await?;
                response.pop();

                AuthData::Google(GoogleAuth {
                    username: input.username.clone(),
                    password: input.password.clone(),
                    auth_token: Some(auth_token.to_string()),
                    post_token: Some(response),
                })
            }
        };
        self.auth_input = auth_input;

        Ok(self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)?
            .clone())
    }

    pub async fn user_info(&self, client: &Client) -> Result<User, ApiError> {
        let response = self
            .get_request(
                "reader/api/0/user-info".to_string(),
                &mut Vec::new(),
                client,
            )
            .await?;
        let user: User = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(user)
    }

    pub async fn unread_count(&self, client: &Client) -> Result<Unread, ApiError> {
        let response = self
            .get_request(
                "reader/api/0/unread-count".to_string(),
                &mut Vec::new(),
                client,
            )
            .await?;
        let unread: Unread = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(unread)
    }

    pub async fn subscription_list(&self, client: &Client) -> Result<Feeds, ApiError> {
        let response = self
            .get_request(
                "reader/api/0/subscription/list".to_string(),
                &mut Vec::new(),
                client,
            )
            .await?;
        let subscriptions: Feeds = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(subscriptions)
    }

    pub async fn subscription_create(
        &self,
        url: &Url,
        name: Option<&str>,
        to_stream: Option<&str>,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut params = vec![
            ("ac".to_string(), "subscribe".to_string()),
            ("s".to_string(), format!("feed/{}", &url.as_str())),
        ];
        if let Some(name) = name {
            params.push(("t".to_string(), name.to_string()));
        }
        if let Some(to_stream) = to_stream {
            params.push(("a".to_string(), to_stream.to_string()));
        }
        let response = self
            .post_request(
                "reader/api/0/subscription/edit".to_string(),
                &mut params,
                None,
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    pub async fn subscription_edit(
        &self,
        item_id: &str,
        name: Option<&str>,
        from_stream: Option<&str>,
        to_stream: Option<&str>,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut params = vec![
            ("ac".to_string(), "edit".to_string()),
            ("s".to_string(), item_id.to_string()),
        ];

        if let Some(name) = name {
            params.push(("t".to_string(), name.to_string()));
        }
        if let Some(from_stream) = from_stream {
            params.push(("r".to_string(), from_stream.to_string()));
        }
        if let Some(to_stream) = to_stream {
            params.push(("a".to_string(), to_stream.to_string()));
        }

        let response = self
            .post_request(
                "reader/api/0/subscription/edit".to_string(),
                &mut params,
                None,
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    pub async fn subscription_delete(
        &self,
        stream_id: &str,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut params = vec![
            ("ac".to_string(), "unsubscribe".to_string()),
            ("s".to_string(), stream_id.to_string()),
        ];

        let response = self
            .post_request(
                "reader/api/0/subscription/edit".to_string(),
                &mut params,
                None,
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    pub async fn subscription_quickadd(
        &self,
        url: &Url,
        client: &Client,
    ) -> Result<QuickFeed, ApiError> {
        let mut params = vec![("quickadd".to_string(), (&url.as_str()).to_string())];
        let response = self
            .post_request(
                "reader/api/0/subscription/quickadd".to_string(),
                &mut params,
                None,
                client,
            )
            .await?;
        let subscriptions: QuickFeed =
            serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(subscriptions)
    }

    // untested
    pub async fn import(&self, opml: String, client: &Client) -> Result<u64, ApiError> {
        let mut params: Vec<(String, String)> = Vec::new();

        match &*self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)?
        {
            AuthData::Inoreader(_) | AuthData::Uninitialized => {}
            AuthData::Google(auth_data) => {
                // check if post token is valid else raise error
                if auth_data.post_token.is_none() {
                    return Err(ApiErrorKind::Token.into());
                }
                let post_token: Option<&str> = auth_data.post_token.as_deref();
                params.push(("T".to_string(), post_token.unwrap().to_string()));
            }
        };

        let api_url: Url = self
            .base_uri
            .join("reader/api/0/subscription/import")
            .context(ApiErrorKind::Url)?;

        let response = client
            .post(api_url.clone())
            .headers(self.get_auth_headers()?)
            .query(&params)
            .body(opml)
            .send()
            .await
            .context(ApiErrorKind::Http)?;

        let status = response.status();
        let response = response.text().await.context(ApiErrorKind::Http)?;
        if status != StatusCode::OK {
            let error: GReaderError =
                serde_json::from_str(&response).context(ApiErrorKind::Json)?;
            error!("GReader API: {}", error.errors.join("; "));
            return Err(ApiErrorKind::GReader(error).into());
        }

        if response.starts_with("OK: ") {
            Ok(response.replace("Ok: ", "").parse::<u64>().unwrap())
        } else {
            let error: GReaderError = GReaderError {
                errors: vec![response],
            };
            Err(ApiErrorKind::GReader(error).into())
        }
    }

    // untested
    pub async fn export(&self, client: &Client) -> Result<String, ApiError> {
        let api_url: Url = self
            .base_uri
            .join("reader/api/0/subscription/export")
            .context(ApiErrorKind::Url)?;

        let response = client
            .get(api_url.clone())
            .headers(self.get_auth_headers()?)
            .send()
            .await
            .context(ApiErrorKind::Http)?;

        let status = response.status();
        let response = response.text().await.context(ApiErrorKind::Http)?;
        if status != StatusCode::OK {
            let error: GReaderError =
                serde_json::from_str(&response).context(ApiErrorKind::Json)?;
            error!("GReader API: {}", error.errors.join("; "));
            return Err(ApiErrorKind::GReader(error).into());
        }
        Ok(response)
    }

    // untested
    #[cfg(feature = "feedhq")]
    pub async fn subscribed(&self, stream_id: &str, client: &Client) -> Result<bool, ApiError> {
        let mut params = vec![("s".to_string(), stream_id.to_string())];
        let response = self
            .get_request("reader/api/0/subscribed".to_string(), &mut params, client)
            .await?;
        match &response[..] {
            "true" => Ok(true),
            "false" => Ok(false),
            _ => {
                let error: GReaderError = GReaderError {
                    errors: vec![response.to_string()],
                };
                Err(ApiErrorKind::GReader(error).into())
            }
        }
    }

    #[allow(clippy::too_many_arguments)]
    pub async fn stream_contents(
        &self,
        stream_id: Option<&str>,
        reverse_order: bool,
        amount: Option<u64>,
        continuation: Option<&str>,
        exclude_stream: Option<&str>,
        include_stream: Option<&str>,
        filter_older: Option<i64>,
        filter_newer: Option<i64>,
        client: &Client,
    ) -> Result<Stream, ApiError> {
        let mut params: Vec<(String, String)> = Vec::new();
        if reverse_order {
            params.push(("r".to_string(), "o".to_string()));
        }
        if let Some(n) = amount {
            params.push(("n".to_string(), n.to_string()))
        }
        if let Some(c) = continuation {
            params.push(("c".to_string(), c.to_string()))
        }
        if let Some(s) = exclude_stream {
            params.push(("xt".to_string(), s.to_string()))
        }
        if let Some(s) = include_stream {
            params.push(("it".to_string(), s.to_string()))
        }
        if let Some(t) = filter_older {
            params.push(("ot".to_string(), t.to_string()))
        }
        if let Some(t) = filter_newer {
            params.push(("nt".to_string(), t.to_string()))
        }

        let query = "reader/api/0/stream/contents";
        let query = if let Some(stream_id) = stream_id {
            format!("{}/{}", query, stream_id)
        } else {
            query.into()
        };
        let response = self.post_request(query, &mut params, None, client).await?;

        let stream: Stream = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(stream)
    }

    #[allow(clippy::too_many_arguments)]
    pub async fn items_ids(
        &self,
        stream_id: Option<&str>,
        amount: Option<u64>,
        include_all_direct_stream_ids: bool,
        continuation: Option<&str>,
        exclude_stream: Option<&str>,
        include_stream: Option<&str>,
        filter_older: Option<i64>,
        filter_newer: Option<i64>,
        client: &Client,
    ) -> Result<ItemRefs, ApiError> {
        let mut params = Vec::new();

        if let Some(amount) = amount {
            params.push(("n".to_string(), amount.to_string()));
        }
        if let Some(stream_id) = stream_id {
            params.push(("s".to_string(), stream_id.to_string()));
        }
        if let Some(c) = continuation {
            params.push(("c".to_string(), c.to_string()));
        }
        if include_all_direct_stream_ids {
            params.push(("includeAllDirectStreamIds".to_string(), "true".to_string()));
        }
        if let Some(s) = exclude_stream {
            params.push(("xt".to_string(), s.to_string()))
        }
        if let Some(s) = include_stream {
            params.push(("it".to_string(), s.to_string()))
        }
        if let Some(t) = filter_older {
            params.push(("ot".to_string(), t.to_string()))
        }
        if let Some(t) = filter_newer {
            params.push(("nt".to_string(), t.to_string()))
        }
        let response = self
            .get_request(
                "reader/api/0/stream/items/ids".to_string(),
                &mut params,
                client,
            )
            .await?;

        let item_refs: ItemRefs = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(item_refs)
    }

    #[cfg(feature = "feedhq")]
    pub async fn items_count(
        &self,
        stream_id: &str,
        get_latest_date: bool,
        client: &Client,
    ) -> Result<String, ApiError> {
        let mut params = vec![("s".to_string(), stream_id.to_string())];
        if get_latest_date {
            params.push(("a".to_string(), "true".to_string()))
        }
        let response = self
            .get_request(
                "reader/api/0/stream/items/count".to_string(),
                &mut params,
                client,
            )
            .await?;
        Ok(response)
    }

    pub async fn items_contents(
        &self,
        item_ids: Vec<String>,
        client: &Client,
    ) -> Result<Stream, ApiError> {
        let mut form_params: Vec<(String, String)> = Vec::new();
        for item_id in item_ids {
            form_params.push(("i".to_string(), item_id.to_string()))
        }

        let response = self
            .post_request(
                "reader/api/0/stream/items/contents".to_string(),
                &mut Vec::new(),
                Some(form_params),
                client,
            )
            .await?;

        let stream: Stream = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(stream)
    }

    pub async fn tag_list(&self, client: &Client) -> Result<Taggings, ApiError> {
        let response = self
            .get_request("reader/api/0/tag/list".to_string(), &mut Vec::new(), client)
            .await?;
        let tags: Taggings = serde_json::from_str(&response).context(ApiErrorKind::Json)?;
        Ok(tags)
    }

    pub async fn tag_delete(
        &self,
        stream_type: StreamType,
        id: &str,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut form_params: Vec<(String, String)> = Vec::new();
        let target: String = String::from(stream_type);
        form_params.push((target, id.to_string()));
        let response = self
            .post_request(
                "reader/api/0/disable-tag".to_string(),
                &mut Vec::new(),
                Some(form_params),
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    pub async fn tag_rename(
        &self,
        stream_type: StreamType,
        old_name: &str,
        new_name: &str,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut form_params: Vec<(String, String)> = Vec::new();
        let target: String = String::from(stream_type);
        form_params.push((target, old_name.to_string()));
        form_params.push(("dest".to_string(), new_name.to_string()));
        let response = self
            .post_request(
                "reader/api/0/rename-tag".to_string(),
                &mut Vec::new(),
                Some(form_params),
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    // TODO have better parameters, since it is not obvious what is add and remove
    pub async fn tag_edit(
        &self,
        item_ids: &[&str],
        tag_add: Option<&str>,
        tag_remove: Option<&str>,
        client: &Client,
    ) -> Result<(), ApiError> {
        if tag_add.is_none() && tag_remove.is_none() {
            return Err(ApiErrorKind::Input.into());
        }

        let mut form_params: Vec<(String, String)> = Vec::new();
        for item_id in item_ids {
            form_params.push(("i".to_string(), item_id.to_string()));
        }
        if let Some(remove) = tag_remove {
            form_params.push(("r".to_string(), remove.to_string()));
        }
        if let Some(add) = tag_add {
            form_params.push(("a".to_string(), add.to_string()));
        }

        let response = self
            .post_request(
                "reader/api/0/edit-tag".to_string(),
                &mut Vec::new(),
                Some(form_params),
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    pub async fn mark_all_as_read(
        &self,
        stream_id: &str,
        older_than: Option<u64>,
        client: &Client,
    ) -> Result<(), ApiError> {
        let mut params = vec![("s".to_string(), stream_id.to_string())];

        if let Some(older_than) = older_than {
            params.push(("ts".to_string(), older_than.to_string()));
        }
        let response = self
            .post_request(
                "reader/api/0/mark-all-as-read".to_string(),
                &mut params,
                None,
                client,
            )
            .await?;
        GReaderApi::chech_ok_response(&response)
    }

    #[allow(unused)]
    #[cfg(any(feature = "feedhq", feature = "oldreader"))]
    pub async fn preference_list(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }
    #[allow(unused)]
    #[cfg(any(feature = "feedhq", feature = "oldreader"))]
    pub async fn preference_stream_list(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }

    #[allow(unused)]
    #[cfg(any(feature = "feedhq", feature = "oldreader"))]
    pub async fn friends_list(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }
    #[allow(unused)]
    #[cfg(feature = "oldreader")]
    pub async fn friends_edit(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }

    #[allow(unused)]
    #[cfg(feature = "inoreader")]
    pub async fn inoreader_refresh_token(
        &self,
        client: &Client,
    ) -> Result<InoreaderAuth, ApiError> {
        let auth_data = match &*self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)?
        {
            AuthData::Inoreader(auth_data) => auth_data.clone(),
            _ => return Err(ApiErrorKind::Token.into()),
        };

        let client_id = auth_data.client_id.clone();
        let client_secret = auth_data.client_secret.clone();
        let refresh_token = auth_data.refresh_token.clone();

        let mut map: HashMap<String, String> = HashMap::new();
        map.insert("client_id".into(), client_id);
        map.insert("client_secret".into(), client_secret);
        map.insert("grant_type".into(), "refresh_token".into());
        map.insert("refresh_token".into(), refresh_token);

        let response = client
            .post("https://www.inoreader.com/oauth2/token")
            .form(&map)
            .send()
            .await
            .context(ApiErrorKind::Http)?
            .text()
            .await
            .context(ApiErrorKind::Http)?;

        let oauth_response = OAuthResponse::from_json(&response).context(ApiErrorKind::Json)?;

        let now = Utc::now();
        let token_expires = now + Duration::seconds(oauth_response.expires_in);

        let inoreader_auth = InoreaderAuth {
            client_id: auth_data.client_id.clone(),
            client_secret: auth_data.client_secret.clone(),
            access_token: oauth_response.access_token,
            refresh_token: oauth_response.refresh_token,
            expires_at: token_expires,
        };

        *self
            .auth
            .lock()
            .map_err(|_| ApiErrorKind::InternalMutabilty)? =
            AuthData::Inoreader(inoreader_auth.clone());

        Ok(inoreader_auth)
    }

    #[allow(unused)]
    #[cfg(feature = "inoreader")]
    pub async fn create_active_search(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }

    #[allow(unused)]
    #[cfg(feature = "inoreader")]
    pub async fn delete_active_search(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }

    #[allow(unused)]
    #[cfg(feature = "oldreader")]
    pub async fn add_comment(&self, client: &Client) -> Result<(), ApiError> {
        unimplemented!();
    }
}
