//! A big dictionary of named colors and their
//! corresponding RGBA values

pub(crate) struct NamedColorMap {
    name_to_rgba: phf::Map<&'static str, [u8; 4]>,
    rgba_to_name: phf::Map<[u8; 3], &'static str>,
}

impl NamedColorMap {
    pub fn get_by_name(&self, name: &str) -> Option<&[u8; 4]> {
        self.name_to_rgba.get(name)
    }

    // transparent is not handled
    pub fn get_by_rgba(&self, rgba: [u8; 3]) -> Option<&&str> {
        self.rgba_to_name.get(&rgba)
    }
}

// Adding one color requires adding to both map due to limitation in phf crate.
pub(crate) static NAMED_COLORS: NamedColorMap = NamedColorMap {
    name_to_rgba: phf::phf_map! {
        "aliceblue" => [0xF0, 0xF8, 0xFF, 0xFF],
        "antiquewhite" => [0xFA, 0xEB, 0xD7, 0xFF],
        "aqua" => [0x00, 0xFF, 0xFF, 0xFF],
        "aquamarine" => [0x7F, 0xFF, 0xD4, 0xFF],
        "azure" => [0xF0, 0xFF, 0xFF, 0xFF],
        "beige" => [0xF5, 0xF5, 0xDC, 0xFF],
        "bisque" => [0xFF, 0xE4, 0xC4, 0xFF],
        "black" => [0x00, 0x00, 0x00, 0xFF],
        "blanchedalmond" => [0xFF, 0xEB, 0xCD, 0xFF],
        "blue" => [0x00, 0x00, 0xFF, 0xFF],
        "blueviolet" => [0x8A, 0x2B, 0xE2, 0xFF],
        "brown" => [0xA5, 0x2A, 0x2A, 0xFF],
        "burlywood" => [0xDE, 0xB8, 0x87, 0xFF],
        "cadetblue" => [0x5F, 0x9E, 0xA0, 0xFF],
        "chartreuse" => [0x7F, 0xFF, 0x00, 0xFF],
        "chocolate" => [0xD2, 0x69, 0x1E, 0xFF],
        "coral" => [0xFF, 0x7F, 0x50, 0xFF],
        "cornflowerblue" => [0x64, 0x95, 0xED, 0xFF],
        "cornsilk" => [0xFF, 0xF8, 0xDC, 0xFF],
        "crimson" => [0xDC, 0x14, 0x3C, 0xFF],
        "darkblue" => [0x00, 0x00, 0x8B, 0xFF],
        "darkcyan" => [0x00, 0x8B, 0x8B, 0xFF],
        "darkgoldenrod" => [0xB8, 0x86, 0x0B, 0xFF],
        "darkgray" => [0xA9, 0xA9, 0xA9, 0xFF],
        "darkgreen" => [0x00, 0x64, 0x00, 0xFF],
        "darkkhaki" => [0xBD, 0xB7, 0x6B, 0xFF],
        "darkmagenta" => [0x8B, 0x00, 0x8B, 0xFF],
        "darkolivegreen" => [0x55, 0x6B, 0x2F, 0xFF],
        "darkorange" => [0xFF, 0x8C, 0x00, 0xFF],
        "darkorchid" => [0x99, 0x32, 0xCC, 0xFF],
        "darkred" => [0x8B, 0x00, 0x00, 0xFF],
        "darksalmon" => [0xE9, 0x96, 0x7A, 0xFF],
        "darkseagreen" => [0x8F, 0xBC, 0x8F, 0xFF],
        "darkslateblue" => [0x48, 0x3D, 0x8B, 0xFF],
        "darkslategray" => [0x2F, 0x4F, 0x4F, 0xFF],
        "darkturquoise" => [0x00, 0xCE, 0xD1, 0xFF],
        "darkviolet" => [0x94, 0x00, 0xD3, 0xFF],
        "deeppink" => [0xFF, 0x14, 0x93, 0xFF],
        "deepskyblue" => [0x00, 0xBF, 0xFF, 0xFF],
        "dimgray" => [0x69, 0x69, 0x69, 0xFF],
        "dodgerblue" => [0x1E, 0x90, 0xFF, 0xFF],
        "firebrick" => [0xB2, 0x22, 0x22, 0xFF],
        "floralwhite" => [0xFF, 0xFA, 0xF0, 0xFF],
        "forestgreen" => [0x22, 0x8B, 0x22, 0xFF],
        "fuchsia" => [0xFF, 0x00, 0xFF, 0xFF],
        "gainsboro" => [0xDC, 0xDC, 0xDC, 0xFF],
        "ghostwhite" => [0xF8, 0xF8, 0xFF, 0xFF],
        "gold" => [0xFF, 0xD7, 0x00, 0xFF],
        "goldenrod" => [0xDA, 0xA5, 0x20, 0xFF],
        "gray" => [0x80, 0x80, 0x80, 0xFF],
        "green" => [0x00, 0x80, 0x00, 0xFF],
        "greenyellow" => [0xAD, 0xFF, 0x2F, 0xFF],
        "honeydew" => [0xF0, 0xFF, 0xF0, 0xFF],
        "hotpink" => [0xFF, 0x69, 0xB4, 0xFF],
        "indianred" => [0xCD, 0x5C, 0x5C, 0xFF],
        "indigo" => [0x4B, 0x00, 0x82, 0xFF],
        "ivory" => [0xFF, 0xFF, 0xF0, 0xFF],
        "khaki" => [0xF0, 0xE6, 0x8C, 0xFF],
        "lavender" => [0xE6, 0xE6, 0xFA, 0xFF],
        "lavenderblush" => [0xFF, 0xF0, 0xF5, 0xFF],
        "lawngreen" => [0x7C, 0xFC, 0x00, 0xFF],
        "lemonchiffon" => [0xFF, 0xFA, 0xCD, 0xFF],
        "lightblue" => [0xAD, 0xD8, 0xE6, 0xFF],
        "lightcoral" => [0xF0, 0x80, 0x80, 0xFF],
        "lightcyan" => [0xE0, 0xFF, 0xFF, 0xFF],
        "lightgoldenrodyellow" => [0xFA, 0xFA, 0xD2, 0xFF],
        "lightgray" => [0xD3, 0xD3, 0xD3, 0xFF],
        "lightgreen" => [0x90, 0xEE, 0x90, 0xFF],
        "lightpink" => [0xFF, 0xB6, 0xC1, 0xFF],
        "lightsalmon" => [0xFF, 0xA0, 0x7A, 0xFF],
        "lightseagreen" => [0x20, 0xB2, 0xAA, 0xFF],
        "lightskyblue" => [0x87, 0xCE, 0xFA, 0xFF],
        "lightslategray" => [0x77, 0x88, 0x99, 0xFF],
        "lightsteelblue" => [0xB0, 0xC4, 0xDE, 0xFF],
        "lightyellow" => [0xFF, 0xFF, 0xE0, 0xFF],
        "lime" => [0x00, 0xFF, 0x00, 0xFF],
        "limegreen" => [0x32, 0xCD, 0x32, 0xFF],
        "linen" => [0xFA, 0xF0, 0xE6, 0xFF],
        "maroon" => [0x80, 0x00, 0x00, 0xFF],
        "mediumaquamarine" => [0x66, 0xCD, 0xAA, 0xFF],
        "mediumblue" => [0x00, 0x00, 0xCD, 0xFF],
        "mediumorchid" => [0xBA, 0x55, 0xD3, 0xFF],
        "mediumpurple" => [0x93, 0x70, 0xDB, 0xFF],
        "mediumseagreen" => [0x3C, 0xB3, 0x71, 0xFF],
        "mediumslateblue" => [0x7B, 0x68, 0xEE, 0xFF],
        "mediumspringgreen" => [0x00, 0xFA, 0x9A, 0xFF],
        "mediumturquoise" => [0x48, 0xD1, 0xCC, 0xFF],
        "mediumvioletred" => [0xC7, 0x15, 0x85, 0xFF],
        "midnightblue" => [0x19, 0x19, 0x70, 0xFF],
        "mintcream" => [0xF5, 0xFF, 0xFA, 0xFF],
        "mistyrose" => [0xFF, 0xE4, 0xE1, 0xFF],
        "moccasin" => [0xFF, 0xE4, 0xB5, 0xFF],
        "navajowhite" => [0xFF, 0xDE, 0xAD, 0xFF],
        "navy" => [0x00, 0x00, 0x80, 0xFF],
        "oldlace" => [0xFD, 0xF5, 0xE6, 0xFF],
        "olive" => [0x80, 0x80, 0x00, 0xFF],
        "olivedrab" => [0x6B, 0x8E, 0x23, 0xFF],
        "orange" => [0xFF, 0xA5, 0x00, 0xFF],
        "orangered" => [0xFF, 0x45, 0x00, 0xFF],
        "orchid" => [0xDA, 0x70, 0xD6, 0xFF],
        "palegoldenrod" => [0xEE, 0xE8, 0xAA, 0xFF],
        "palegreen" => [0x98, 0xFB, 0x98, 0xFF],
        "paleturquoise" => [0xAF, 0xEE, 0xEE, 0xFF],
        "palevioletred" => [0xDB, 0x70, 0x93, 0xFF],
        "papayawhip" => [0xFF, 0xEF, 0xD5, 0xFF],
        "peachpuff" => [0xFF, 0xDA, 0xB9, 0xFF],
        "peru" => [0xCD, 0x85, 0x3F, 0xFF],
        "pink" => [0xFF, 0xC0, 0xCB, 0xFF],
        "plum" => [0xDD, 0xA0, 0xDD, 0xFF],
        "powderblue" => [0xB0, 0xE0, 0xE6, 0xFF],
        "purple" => [0x80, 0x00, 0x80, 0xFF],
        "rebeccapurple" => [0x66, 0x33, 0x99, 0xFF],
        "red" => [0xFF, 0x00, 0x00, 0xFF],
        "rosybrown" => [0xBC, 0x8F, 0x8F, 0xFF],
        "royalblue" => [0x41, 0x69, 0xE1, 0xFF],
        "saddlebrown" => [0x8B, 0x45, 0x13, 0xFF],
        "salmon" => [0xFA, 0x80, 0x72, 0xFF],
        "sandybrown" => [0xF4, 0xA4, 0x60, 0xFF],
        "seagreen" => [0x2E, 0x8B, 0x57, 0xFF],
        "seashell" => [0xFF, 0xF5, 0xEE, 0xFF],
        "sienna" => [0xA0, 0x52, 0x2D, 0xFF],
        "silver" => [0xC0, 0xC0, 0xC0, 0xFF],
        "skyblue" => [0x87, 0xCE, 0xEB, 0xFF],
        "slateblue" => [0x6A, 0x5A, 0xCD, 0xFF],
        "slategray" => [0x70, 0x80, 0x90, 0xFF],
        "snow" => [0xFF, 0xFA, 0xFA, 0xFF],
        "springgreen" => [0x00, 0xFF, 0x7F, 0xFF],
        "steelblue" => [0x46, 0x82, 0xB4, 0xFF],
        "tan" => [0xD2, 0xB4, 0x8C, 0xFF],
        "teal" => [0x00, 0x80, 0x80, 0xFF],
        "thistle" => [0xD8, 0xBF, 0xD8, 0xFF],
        "tomato" => [0xFF, 0x63, 0x47, 0xFF],
        "turquoise" => [0x40, 0xE0, 0xD0, 0xFF],
        "violet" => [0xEE, 0x82, 0xEE, 0xFF],
        "wheat" => [0xF5, 0xDE, 0xB3, 0xFF],
        "white" => [0xFF, 0xFF, 0xFF, 0xFF],
        "whitesmoke" => [0xF5, 0xF5, 0xF5, 0xFF],
        "yellow" => [0xFF, 0xFF, 0x00, 0xFF],
        "yellowgreen" => [0x9A, 0xCD, 0x32, 0xFF],
        "transparent" => [0x00, 0x00, 0x00, 0x00],
    },
    rgba_to_name: phf::phf_map! {
        [0xF0, 0xF8, 0xFF] => "aliceblue",
        [0xFA, 0xEB, 0xD7] => "antiquewhite",
        [0x00, 0xFF, 0xFF] => "aqua",
        [0x7F, 0xFF, 0xD4] => "aquamarine",
        [0xF0, 0xFF, 0xFF] => "azure",
        [0xF5, 0xF5, 0xDC] => "beige",
        [0xFF, 0xE4, 0xC4] => "bisque",
        [0x00, 0x00, 0x00] => "black",
        [0xFF, 0xEB, 0xCD] => "blanchedalmond",
        [0x00, 0x00, 0xFF] => "blue",
        [0x8A, 0x2B, 0xE2] => "blueviolet",
        [0xA5, 0x2A, 0x2A] => "brown",
        [0xDE, 0xB8, 0x87] => "burlywood",
        [0x5F, 0x9E, 0xA0] => "cadetblue",
        [0x7F, 0xFF, 0x00] => "chartreuse",
        [0xD2, 0x69, 0x1E] => "chocolate",
        [0xFF, 0x7F, 0x50] => "coral",
        [0x64, 0x95, 0xED] => "cornflowerblue",
        [0xFF, 0xF8, 0xDC] => "cornsilk",
        [0xDC, 0x14, 0x3C] => "crimson",
        [0x00, 0x00, 0x8B] => "darkblue",
        [0x00, 0x8B, 0x8B] => "darkcyan",
        [0xB8, 0x86, 0x0B] => "darkgoldenrod",
        [0xA9, 0xA9, 0xA9] => "darkgray",
        [0x00, 0x64, 0x00] => "darkgreen",
        [0xBD, 0xB7, 0x6B] => "darkkhaki",
        [0x8B, 0x00, 0x8B] => "darkmagenta",
        [0x55, 0x6B, 0x2F] => "darkolivegreen",
        [0xFF, 0x8C, 0x00] => "darkorange",
        [0x99, 0x32, 0xCC] => "darkorchid",
        [0x8B, 0x00, 0x00] => "darkred",
        [0xE9, 0x96, 0x7A] => "darksalmon",
        [0x8F, 0xBC, 0x8F] => "darkseagreen",
        [0x48, 0x3D, 0x8B] => "darkslateblue",
        [0x2F, 0x4F, 0x4F] => "darkslategray",
        [0x00, 0xCE, 0xD1] => "darkturquoise",
        [0x94, 0x00, 0xD3] => "darkviolet",
        [0xFF, 0x14, 0x93] => "deeppink",
        [0x00, 0xBF, 0xFF] => "deepskyblue",
        [0x69, 0x69, 0x69] => "dimgray",
        [0x1E, 0x90, 0xFF] => "dodgerblue",
        [0xB2, 0x22, 0x22] => "firebrick",
        [0xFF, 0xFA, 0xF0] => "floralwhite",
        [0x22, 0x8B, 0x22] => "forestgreen",
        [0xFF, 0x00, 0xFF] => "fuchsia",
        [0xDC, 0xDC, 0xDC] => "gainsboro",
        [0xF8, 0xF8, 0xFF] => "ghostwhite",
        [0xFF, 0xD7, 0x00] => "gold",
        [0xDA, 0xA5, 0x20] => "goldenrod",
        [0x80, 0x80, 0x80] => "gray",
        [0x00, 0x80, 0x00] => "green",
        [0xAD, 0xFF, 0x2F] => "greenyellow",
        [0xF0, 0xFF, 0xF0] => "honeydew",
        [0xFF, 0x69, 0xB4] => "hotpink",
        [0xCD, 0x5C, 0x5C] => "indianred",
        [0x4B, 0x00, 0x82] => "indigo",
        [0xFF, 0xFF, 0xF0] => "ivory",
        [0xF0, 0xE6, 0x8C] => "khaki",
        [0xE6, 0xE6, 0xFA] => "lavender",
        [0xFF, 0xF0, 0xF5] => "lavenderblush",
        [0x7C, 0xFC, 0x00] => "lawngreen",
        [0xFF, 0xFA, 0xCD] => "lemonchiffon",
        [0xAD, 0xD8, 0xE6] => "lightblue",
        [0xF0, 0x80, 0x80] => "lightcoral",
        [0xE0, 0xFF, 0xFF] => "lightcyan",
        [0xFA, 0xFA, 0xD2] => "lightgoldenrodyellow",
        [0xD3, 0xD3, 0xD3] => "lightgray",
        [0x90, 0xEE, 0x90] => "lightgreen",
        [0xFF, 0xB6, 0xC1] => "lightpink",
        [0xFF, 0xA0, 0x7A] => "lightsalmon",
        [0x20, 0xB2, 0xAA] => "lightseagreen",
        [0x87, 0xCE, 0xFA] => "lightskyblue",
        [0x77, 0x88, 0x99] => "lightslategray",
        [0xB0, 0xC4, 0xDE] => "lightsteelblue",
        [0xFF, 0xFF, 0xE0] => "lightyellow",
        [0x00, 0xFF, 0x00] => "lime",
        [0x32, 0xCD, 0x32] => "limegreen",
        [0xFA, 0xF0, 0xE6] => "linen",
        [0x80, 0x00, 0x00] => "maroon",
        [0x66, 0xCD, 0xAA] => "mediumaquamarine",
        [0x00, 0x00, 0xCD] => "mediumblue",
        [0xBA, 0x55, 0xD3] => "mediumorchid",
        [0x93, 0x70, 0xDB] => "mediumpurple",
        [0x3C, 0xB3, 0x71] => "mediumseagreen",
        [0x7B, 0x68, 0xEE] => "mediumslateblue",
        [0x00, 0xFA, 0x9A] => "mediumspringgreen",
        [0x48, 0xD1, 0xCC] => "mediumturquoise",
        [0xC7, 0x15, 0x85] => "mediumvioletred",
        [0x19, 0x19, 0x70] => "midnightblue",
        [0xF5, 0xFF, 0xFA] => "mintcream",
        [0xFF, 0xE4, 0xE1] => "mistyrose",
        [0xFF, 0xE4, 0xB5] => "moccasin",
        [0xFF, 0xDE, 0xAD] => "navajowhite",
        [0x00, 0x00, 0x80] => "navy",
        [0xFD, 0xF5, 0xE6] => "oldlace",
        [0x80, 0x80, 0x00] => "olive",
        [0x6B, 0x8E, 0x23] => "olivedrab",
        [0xFF, 0xA5, 0x00] => "orange",
        [0xFF, 0x45, 0x00] => "orangered",
        [0xDA, 0x70, 0xD6] => "orchid",
        [0xEE, 0xE8, 0xAA] => "palegoldenrod",
        [0x98, 0xFB, 0x98] => "palegreen",
        [0xAF, 0xEE, 0xEE] => "paleturquoise",
        [0xDB, 0x70, 0x93] => "palevioletred",
        [0xFF, 0xEF, 0xD5] => "papayawhip",
        [0xFF, 0xDA, 0xB9] => "peachpuff",
        [0xCD, 0x85, 0x3F] => "peru",
        [0xFF, 0xC0, 0xCB] => "pink",
        [0xDD, 0xA0, 0xDD] => "plum",
        [0xB0, 0xE0, 0xE6] => "powderblue",
        [0x80, 0x00, 0x80] => "purple",
        [0x66, 0x33, 0x99] => "rebeccapurple",
        [0xFF, 0x00, 0x00] => "red",
        [0xBC, 0x8F, 0x8F] => "rosybrown",
        [0x41, 0x69, 0xE1] => "royalblue",
        [0x8B, 0x45, 0x13] => "saddlebrown",
        [0xFA, 0x80, 0x72] => "salmon",
        [0xF4, 0xA4, 0x60] => "sandybrown",
        [0x2E, 0x8B, 0x57] => "seagreen",
        [0xFF, 0xF5, 0xEE] => "seashell",
        [0xA0, 0x52, 0x2D] => "sienna",
        [0xC0, 0xC0, 0xC0] => "silver",
        [0x87, 0xCE, 0xEB] => "skyblue",
        [0x6A, 0x5A, 0xCD] => "slateblue",
        [0x70, 0x80, 0x90] => "slategray",
        [0xFF, 0xFA, 0xFA] => "snow",
        [0x00, 0xFF, 0x7F] => "springgreen",
        [0x46, 0x82, 0xB4] => "steelblue",
        [0xD2, 0xB4, 0x8C] => "tan",
        [0x00, 0x80, 0x80] => "teal",
        [0xD8, 0xBF, 0xD8] => "thistle",
        [0xFF, 0x63, 0x47] => "tomato",
        [0x40, 0xE0, 0xD0] => "turquoise",
        [0xEE, 0x82, 0xEE] => "violet",
        [0xF5, 0xDE, 0xB3] => "wheat",
        [0xFF, 0xFF, 0xFF] => "white",
        [0xF5, 0xF5, 0xF5] => "whitesmoke",
        [0xFF, 0xFF, 0x00] => "yellow",
        [0x9A, 0xCD, 0x32] => "yellowgreen",
    },
};
