pub use node_mut::*;

mod node_mut;

use crate::tree::{NodeId, NodeRef, Tree};
use imbl::HashMap;
use std::hash::Hash;

#[derive(Clone)]
/// A node entry in a `MapTree`
///
/// This consists of a value and the associated key
pub struct Entry<Key: Hash + Eq + Clone, T: Clone> {
    value: T,
    key: Key,
}

impl<Key: Hash + Eq + Clone, T: Clone> Entry<Key, T> {
    pub(crate) fn new(key: Key, value: T) -> Self {
        Self { value, key }
    }

    pub fn value(&self) -> &T {
        &self.value
    }

    pub fn key(&self) -> &Key {
        &self.key
    }
}

/// A persistent, indexable, hierarchical data structure that maintains a key-node mapping
///
/// A [tree](https://en.wikipedia.org/wiki/Tree_(data_structure)) is a set of items organized in a hierarchy. `MapTree` always has at least one item, the root node. The root node may have children nodes, and those nodes may have children of their own – this goes down as deep as you need!
///
/// Every node has exactly one parent, except for the root node, which has no parent.
///
/// A node's children is logically a list – so the order of the children is specified, and you can change it if you want.
///
/// In addition to the hierarchy, `MapTree` maintains a mapping of arbitrary keys (which you specify) to nodes. Every node has a key – you can directly retrieve a node if you know its key. If you don't need this mapping, and would rather have auto-generated IDs, you can use the `Tree` instead.
pub struct MapTree<Key: Hash + Eq + Clone, T: Clone> {
    tree: Tree<Entry<Key, T>>,
    map: HashMap<Key, NodeId>,
}

impl<Key: Hash + Eq + Clone, T: Clone> MapTree<Key, T> {
    /// Create a new Tree with the specified root node key and value
    pub fn new_with_root(key: Key, value: T) -> Self {
        let tree = Tree::new_with_root(Entry::new(key.clone(), value));
        Self {
            map: HashMap::unit(key, tree.root().id()),
            tree,
        }
    }

    /// Verifies that the Tree is in a valid state
    ///
    /// All Trees must be valid; an invalid tree indicates a bug in the library code
    ///
    /// Checks that:
    /// - The underlying arena is valid
    /// - There are no orphan nodes
    /// - All references are consistent (e.g. previous_sibling/next_sibling point to each other)
    /// - All references are valid (e.g. next_sibling doesn't point to non-existent node)
    /// - There are no loops
    /// - The root has no siblings
    ///
    /// Panics if the state is invalid
    pub fn validate(&self) {
        self.tree.validate()
    }

    /// Get a reference to the root node
    ///
    /// The tree will always have a root node
    pub fn root(&self) -> NodeRef<Entry<Key, T>> {
        self.tree.root()
    }

    /// Get a mutable reference to the root node
    ///
    /// The tree will always have a root node
    pub fn root_mut(&mut self) -> NodeMut<Key, T> {
        NodeMut::new(self.tree.root_mut(), &mut self.map)
    }

    pub fn get(&self, key: &Key) -> Option<NodeRef<Entry<Key, T>>> {
        let id = self.map.get(key)?;
        self.tree.get(*id)
    }

    pub fn get_mut(&mut self, key: &Key) -> Option<NodeMut<Key, T>> {
        let id = self.map.get(key)?;
        let inner = self.tree.get_mut(*id)?;
        Some(NodeMut::new(inner, &mut self.map))
    }
}
