use crate::map_tree::Entry;
use crate::tree::{Children, NodeId};
use crate::tree_location::Location;
use crate::{tree, CannotRemoveRoot, MoveError};
use imbl::HashMap;
use std::hash::Hash;

/// A mutable reference to a node in a `MapTree`
///
/// The library guarantees that this references a valid node in the associated tree
pub struct NodeMut<'a, Key: Hash + Eq + Clone, T: Clone> {
    inner: tree::NodeMut<'a, Entry<Key, T>>,
    map: &'a mut HashMap<Key, NodeId>,
}

impl<'a, Key: Hash + Eq + Clone, T: Clone> NodeMut<'a, Key, T> {
    pub(crate) fn new(
        inner: tree::NodeMut<'a, Entry<Key, T>>,
        map: &'a mut HashMap<Key, NodeId>,
    ) -> Self {
        Self { inner, map }
    }
}

impl<'a, Key: Hash + Eq + Clone, T: Clone> NodeMut<'a, Key, T> {
    /// Get the entry associated with the node
    pub fn data(&self) -> &Entry<Key, T> {
        &self.inner.data()
    }

    /// Get the node's ID
    pub fn id(&self) -> NodeId {
        self.inner.id()
    }

    /// Iterate over the node's children
    pub fn children(&self) -> Children<Entry<Key, T>> {
        self.inner.children()
    }
}

impl<'a, Key: Hash + Eq + Clone, T: Clone> NodeMut<'a, Key, T> {
    /// Get a mutable reference to the node's data
    pub fn data_mut(&mut self) -> &mut T {
        &mut self.inner.data_mut().value
    }

    /// Push a new child node to the front of the children list
    ///
    /// This operation should always succeed
    ///
    /// (The specified value will become the first child, and any subsequent children will be pushed down)
    pub fn push_front_child(&mut self, key: Key, value: T) -> NodeMut<Key, T> {
        let child = self.inner.push_front_child(Entry::new(key.clone(), value));
        self.map.insert(key, child.id());

        NodeMut::new(child, self.map)
    }

    /// Push a new child as the next sibling of this node
    ///
    /// This is allowed for all nodes except for the root node. When attempting to add a sibling to the root node, this will return None, and nothing will be added to the tree
    pub fn push_next_sibling(&mut self, key: Key, value: T) -> Option<NodeMut<Key, T>> {
        if let Some(child) = self.inner.push_next_sibling(Entry::new(key.clone(), value)) {
            self.map.insert(key, child.id());
            Some(NodeMut::new(child, self.map))
        } else {
            None
        }
    }

    /// Remove this node and all of its descendants from the tree
    ///
    /// Note: if you'd like to do something with the removed values, use `remove_with_consumer` instead.
    ///
    /// Removing the root node is not allowed and will return an error with `CannotRemoveRoot`. All other removals are allowed.
    pub fn remove(self) -> Result<(), CannotRemoveRoot> {
        self.remove_with_consumer(|_| {})
    }

    /// Remove this node and all of its descendants from the tree
    ///
    /// All removed values will be passed to the specified `consumer` function
    ///
    /// Removing the root node is not allowed and will return an error with `CannotRemoveRoot`. All other removals are allowed.
    pub fn remove_with_consumer(
        self,
        mut consumer: impl FnMut(Entry<Key, T>),
    ) -> Result<(), CannotRemoveRoot> {
        self.inner.remove_with_consumer(|entry| {
            self.map.remove(entry.key());
            (consumer)(entry)
        })
    }

    /// Move this node, along with all descendants, to a new location
    ///
    /// If this is not allowed, a `MoveError` will be returned, and no changes will be made. See the `MoveError` documentation for details on possible errors.
    ///
    /// Moving a node to `AfterSibling(self ID)` is allowed and has no effect.
    pub fn move_to(&mut self, location: Location<Key>) -> Result<(), MoveError<Key>> {
        let mapped_location = location
            .into_mapped(|key| {
                self.map
                    .get(&key)
                    .cloned()
                    .ok_or(MoveError::NoSuchNode(key))
            })
            .lift_result()?;

        self.inner
            .move_to(mapped_location)
            .map_err(|err| match err {
                MoveError::NoSuchNode(_) => {
                    unreachable!("node found in map and must exist")
                }
                MoveError::RootCannotHaveSiblings => MoveError::RootCannotHaveSiblings,
                MoveError::CannotMoveUnderSelf => MoveError::CannotMoveUnderSelf,
            })
    }
}
