/// Describes a location in a tree
pub enum Location<Id> {
    /// The next sibling of the specified node
    ///
    /// Inserting a node here would make that node the next sibling after the specified node, and push any further children down
    ///
    /// Note: the specified node should not be the root node, as the root may have no siblings
    AfterSibling(Id),
    /// The first child of the specified node
    ///
    /// Inserting a node here would make that node the first child, and push any other children down
    FirstChildOf(Id),
}

impl<Id> Location<Id> {
    /// Map the inner ID type of the Location
    pub fn into_mapped<NewId>(self, mapper: impl FnOnce(Id) -> NewId) -> Location<NewId> {
        match self {
            Location::AfterSibling(id) => Location::AfterSibling((mapper)(id)),
            Location::FirstChildOf(id) => Location::FirstChildOf((mapper)(id)),
        }
    }
}

impl<Id, E> Location<Result<Id, E>> {
    /// Convert a `Location<Result<Id, E>>` into a `Result<Location<Id>, E>`
    pub fn lift_result(self) -> Result<Location<Id>, E> {
        Ok(match self {
            Location::AfterSibling(id) => Location::AfterSibling(id?),
            Location::FirstChildOf(id) => Location::FirstChildOf(id?),
        })
    }
}
