//! Several functions to group definitions by a certain criteria.

use crate::tl::{Category, Definition, Type};
use std::collections::HashMap;

/// Group the input vector by namespace, filtering by a certain category.
pub(crate) fn group_by_ns(
    definitions: &[Definition],
    category: Category,
) -> HashMap<String, Vec<&Definition>> {
    let mut result = HashMap::new();
    definitions
        .iter()
        .filter(|d| d.category == category)
        .for_each(|d| {
            // We currently only handle zero or one namespace.
            assert!(d.namespace.len() <= 1);
            let ns = d.namespace.get(0).map(|x| &x[..]).unwrap_or("");
            result.entry(ns.into()).or_insert_with(Vec::new).push(d);
        });

    for (_, vec) in result.iter_mut() {
        vec.sort_by_key(|d| &d.name);
    }
    result
}

/// Similar to `group_by_ns`, but for the definition types.
pub(crate) fn group_types_by_ns(definitions: &[Definition]) -> HashMap<Option<String>, Vec<&Type>> {
    let mut result = HashMap::new();
    definitions
        .iter()
        .filter(|d| d.category == Category::Types && !d.ty.generic_ref)
        .for_each(|d| {
            // We currently only handle zero or one namespace.
            assert!(d.namespace.len() <= 1);
            result
                .entry(d.namespace.get(0).map(Clone::clone))
                .or_insert_with(Vec::new)
                .push(&d.ty);
        });

    for (_, vec) in result.iter_mut() {
        vec.sort_by_key(|t| &t.name);
        vec.dedup();
    }
    result
}
