use std::{cmp::Ordering, fmt};

use crate::DebugLisp;

/// A simple representation of the location of some substring within a larger string.
/// Primarily used by [`SpannedError`] to provide user-friendly error formatting.
///
/// [`SpannedError`]: crate::error::SpannedError
#[derive(Clone, Copy, Default, PartialOrd, Ord, PartialEq, Eq, Hash)]
pub struct Span {
	pub start: Position,
	pub end: Position,
}

/// Represents a cursor position within a string. Line and character offsets are
/// zero-indexed in the internal representation, but lines will be printed as 1-indexed by
/// [`SpannedError`] for consistency with IDEs. Character offsets are relative to the
/// current line.
///
/// [`SpannedError`]: crate::error::SpannedError
#[derive(Clone, Copy, Default, PartialEq, Eq, Hash)]
pub struct Position {
	pub line: usize,
	pub character: usize,
}

impl PartialOrd for Position {
	fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
		if self.line == other.line {
			Some(self.character.cmp(&other.character))
		} else {
			Some(self.line.cmp(&other.line))
		}
	}
}

impl Ord for Position {
	fn cmp(&self, other: &Self) -> Ordering {
		self.partial_cmp(other).unwrap()
	}
}

pub trait Spanned {
	fn span(&self) -> Span;
}

#[macro_export]
macro_rules! span {
	($start_line:literal:$start_char:literal...$end_line:literal:$end_char:literal) => {
		::gramatika::Span::new(($start_line, $start_char), ($end_line, $end_char))
	};
}

impl Span {
	pub fn new(start: (usize, usize), end: (usize, usize)) -> Self {
		Self {
			start: Position {
				line: start.0,
				character: start.1,
			},
			end: Position {
				line: end.0,
				character: end.1,
			},
		}
	}

	pub fn through(self, other: Span) -> Span {
		Span {
			start: self.start,
			end: other.end,
		}
	}

	pub fn contains(&self, other: Span) -> bool {
		self.start <= other.start && self.end >= other.end
	}
}

impl fmt::Debug for Span {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{:?}...{:?}", self.start, self.end)
	}
}

impl DebugLisp for Span {
	fn fmt(&self, f: &mut fmt::Formatter<'_>, _: usize) -> fmt::Result {
		fmt::Debug::fmt(self, f)
	}
}

impl fmt::Debug for Position {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}:{}", self.line + 1, self.character + 1)
	}
}
