// Filename: instruction.rs
// Version:	 0.8
// Date:	 18-11-2021 (DD-MM-YYYY)
// Library:  gpcas_isa
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Contains data structs and enums used by the emulator to send instructions to the simulator.

/// The maximum number of register inputs that is needed by any ISA.
pub const MAX_INPUTS: usize = 4;
/// This flag gets set in the register indices. It indicates that the index is for a vector
/// register and therefore might be using a separate register set.
pub const IS_VECTOR_REGISTER: u8 = 0x20;

/// An emulated instruction from the program of the simulation.
///
/// The emulator frontend sends one `Instruction` for each instruction it emulates. The simulator
/// backend then uses it to do timing simulation.
#[derive(Debug, Default, Clone)]
pub struct Instruction {
    /// The address of the instruction.
    pub address: usize,
    /// Contains the address of the memory operand, if it has one.
    pub memory_address: usize,
    /// The target address of the branch, provided the instruction is one.
    pub branch_address: usize,
    /// An ID for the instruction class, see `instruction_type` in the gpcas_base crate. Determines
    /// execution time in the ALU, as well as possible restrictions for execution pipe choice in the
    /// model.
    pub instr_type: u16,
    /// Contains the size of the memory operand in bytes.
    pub memory_operand_size: u16,
    /// Bitflags for different properties of the instruction, see [`instruction_flags`].
    pub flags: u16,
    /// The register indices of the inputs. The full input count may not be needed by all ISAs.
    ///
    /// Registers used as input operands are saved after ones for memory operand calculation.
    pub inputs: [u8; MAX_INPUTS],
    /// How many register inputs the instruction uses in total.
    pub input_register_count: u8,
    /// How many register inputs are used for memory operand calculation.
    pub memory_register_count: u8,
    /// The register index of the output if the instruction has one.
    pub output: u8,
    /// Size of the instruction in bytes.
    pub size: u8,
}

/// Contains constants for control information for the cpu simulation.
///
/// This module contains the flag constants used in [`Instruction::flags`].
pub mod instruction_flags {
    /// The branch is taken.
    ///
    /// This should only be checked if the branch type is [`BR_TYPE_CONDITIONAL`].
    pub const BR_TAKEN: u16 = 0x0001;
    /// The branch is a return from a call.
    ///
    /// This should only be checked if the branch type is [`BR_TYPE_CALL`].
    pub const BR_RETURN: u16 = BR_TAKEN;
    /// The branch uses registers or a memory operand to determine the target address.
    ///
    /// If this is set, the target address might get predicted, but not resolved in the front end of
    /// the front end of the model pipeline.
    pub const BR_INDIRECT: u16 = 0x0002;
    /// The instruction is no branch of any form.
    pub const BR_TYPE_NONE: u16 = 0x0000;
    /// The instruction is a unconditional jump.
    pub const BR_TYPE_JUMP: u16 = 0x0004;
    /// The instruction is a conditional branch.
    pub const BR_TYPE_CONDITIONAL: u16 = 0x0008;
    /// The instruction is either a call or a return instruction.
    pub const BR_TYPE_CALL: u16 = 0x000C;
    /// The bitmask for the branch type. Use this to check the type.
    pub const BR_TYPE_MASK: u16 = 0x000C;

    /// The instruction has a memory input.
    pub const MEM_IN: u16 = 0x0010;
    /// The instruction has a memory output.
    pub const MEM_OUT: u16 = 0x0020;
    /// The instruction has a register output.
    pub const REG_OUT: u16 = 0x0040;
    /// The memory operand is a vector.
    ///
    /// This is used for special vector treatment in case it is necessary.
    pub const MEM_VECTOR: u16 = 0x0080;

    /// The instruction is a micro-operation of a more complex instruction and has another
    /// micro-operation of the same instruction following after it.
    pub const HAS_FOLLOW_UP: u16 = 0x0100;
    /// The instruction switches the execution context, for example when executing a system call.
    /// This can have different effects on the hardware which could be simulated as flushing the L1
    /// caches, for example.
    pub const CONTEXT_CHANGE: u16 = 0x0200;
    /// The simulator should terminate after this instruction.
    ///
    /// Currently there is no difference between a normal exit, a panic or other exceptions.
    pub const TERMINATE: u16 = 0x8000;
}
