// Filename: emulator.rs
// Version:	 0.8
// Date:	 25-01-2022 (DD-MM-YYYY)
// Library:  gpcas_isa
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Contains the Emulator trait as well as a struct for storing variable-length operands.

use crate::instruction::Instruction;

use std::convert::TryInto;
use std::str::FromStr;

/// Can be used to emulate programs by the simulator.
///
/// It is also used to emulate instructions that deviate from the correct program flow. Instructions
/// mustn't retire and commit their result in this case.
pub trait Emulator {
    /// Emulates the instruction at the given address. Returns `None` if program execution
    /// terminated.
    ///
    /// Non-committed instructions mustn't change the emulator state.
    fn emulate_instruction(&mut self, address: usize, commit: bool) -> Option<Instruction>;

    /// Emulates the next instruction in program order. Returns `None` if program execution
    /// terminated.
    fn emulate_next_instruction(&mut self) -> Option<Instruction> {
        self.emulate_instruction(self.get_current_ip(), true)
    }

    /// Returns the number of emulated instructions, regardless of whether they have been committed.
    fn get_instruction_count(&self) -> usize;
    /// Returns the current instruction pointer address.
    fn get_current_ip(&self) -> usize;
    /// Returns the output the emulated program has written to stdout.
    fn get_printed_output(&self) -> &str;
    /// Returns the memory address of the program entry point.
    fn get_start_address(&self) -> usize;
}

/// Additional functions used for interactive emulation.
pub trait InteractiveEmulator: Emulator {
    /// Read access to the entire memory of the emulated application.
    fn get_memory(&self) -> &[u8];
    /// Overwrite a block of memory with new content.
    fn set_memory(&mut self, offset: usize, data: &[u8]);

    /// Get the value of a general purpose register, extended to the native host's integer size.
    fn get_register(&self, index: usize) -> usize;
    /// Set a general purpose register to a new value. Might get truncated to the guest's integer
    /// size.
    fn set_register(&mut self, index: usize, value: usize);

    /// Read access to a vector register.
    ///
    /// The length of the returned slice is dependent on the ISA. ISAs without vector support
    /// should return a slice with zero length.
    fn get_vector_register(&self, index: usize) -> &[u8];
    /// Get a vector register to a new value.
    ///
    /// Slices shorter than the entire vector should cause the additional upper part of the vector
    /// to be set to zero. ISAs with variable vector length support should set the register length
    /// to the length of the value slice.
    /// As an emulator should not call this for ISAs without vector support, the method is allowed
    /// to panic in this case.
    fn set_vector_register(&mut self, index: usize, value: &[u8]);

    /// Emulate a directly input instruction.
    ///
    /// The default implementation works by overwriting the beginning of the emulated application's
    /// memory, emulating from there, and writing the original data back. Because of this
    /// behaviour, the instruction to be emulated should not be dependent on said memory. If you
    /// want to lift this restriction, provide a custom implementation.
    fn emulate_custom_input(&mut self, instruction: &[u8]) {
        let memory_saved = Vec::from(&self.get_memory()[..instruction.len()]);
        self.set_memory(0, instruction);
        self.emulate_instruction(0, true);
        self.set_memory(0, &memory_saved[..]);
    }
}

/// Provides a continuous memory storage for multiple operands.
pub struct OperandStorage {
    /// The actual memory storage.
    memory: Box<[u8]>,
    /// The size of each operand in bytes.
    operand_size: usize,
    /// The number of operands.
    count: usize,
}

/// Formats a c-style string into a rust-style one.
///
/// This works like the `format!` macro, but with C syntax and it does the formatting dynamically at
/// runtime.
///
/// # Example
///
/// ```
/// let number: u64 = 42;
/// let c_string = gpcas_isa::format_c_string("The answer is %4u!", &number.to_le_bytes()[..]);
///
/// assert_eq!(c_string, Ok("The answer is   42!".to_string()));
/// ```
pub fn format_c_string(format_string: &str, parameters: &[u8]) -> Result<String, std::fmt::Error> {
    enum State {
        ReadingText,
        ReadingArgument,
        SetLength(usize),
    }

    let mut output_string = String::with_capacity(format_string.len());
    let mut parameter_offset = 0;
    let mut state = State::ReadingText;

    for (index, character) in format_string.char_indices() {
        state = match state {
            State::ReadingText => {
                if character == '%' {
                    State::ReadingArgument
                } else {
                    output_string.push(character);
                    State::ReadingText
                }
            }
            State::ReadingArgument => {
                if character == '%' {
                    output_string.push('%');
                    State::ReadingText
                } else if let '0'..='9' = character {
                    State::SetLength(index)
                } else {
                    if parameters.len() - parameter_offset < 8 {
                        return Err(std::fmt::Error);
                    }
                    let raw_value = u64::from_le_bytes(
                        parameters[parameter_offset..parameter_offset + 8]
                            .try_into()
                            .unwrap(),
                    );
                    parameter_offset += 8;
                    output_string.push_str(
                        match character {
                            'u' => {
                                format!("{}", raw_value)
                            }
                            'i' | 'd' => {
                                format!("{}", raw_value as i64)
                            }
                            'f' | 'F' => {
                                format!("{}", f64::from_bits(raw_value))
                            }
                            _ => return Err(std::fmt::Error),
                        }
                        .as_str(),
                    );
                    State::ReadingText
                }
            }
            State::SetLength(start) => {
                if let '0'..='9' = character {
                    State::SetLength(start)
                } else {
                    if parameters.len() - parameter_offset < 8 {
                        return Err(std::fmt::Error);
                    }
                    let length = usize::from_str(&format_string[start..index]).unwrap();
                    let raw_value = u64::from_le_bytes(
                        parameters[parameter_offset..parameter_offset + 8]
                            .try_into()
                            .unwrap(),
                    );
                    parameter_offset += 8;
                    let parsed_string = match character {
                        'u' => {
                            format!("{}", raw_value)
                        }
                        'i' | 'd' => {
                            format!("{}", raw_value as i64)
                        }
                        'f' | 'F' => {
                            format!("{}", f64::from_bits(raw_value))
                        }
                        _ => return Err(std::fmt::Error),
                    };
                    for _ in parsed_string.len()..length {
                        output_string.push(' ');
                    }
                    output_string.push_str(parsed_string.as_str());
                    State::ReadingText
                }
            }
        }
    }
    Ok(output_string)
}

impl OperandStorage {
    /// Creates a new `OperandStorage` with `count` operands and a maximum operand size of
    /// `operand_size` bytes. These values can't be changed later.
    pub fn new(operand_size: usize, count: usize) -> Self {
        debug_assert!(operand_size > 0);
        debug_assert!(count > 0);
        OperandStorage {
            memory: vec![0; operand_size * count].into_boxed_slice(),
            operand_size,
            count,
        }
    }

    /// Returns the maximum operand size in bytes.
    #[inline]
    pub fn operand_size(&self) -> usize {
        self.operand_size
    }

    /// Clears all operands.
    #[inline]
    pub fn clear(&mut self) {
        self.memory.fill(0);
    }

    /// Broadcasts the `value` slice over the entire operand at `index`, repeating its value.
    #[inline]
    pub fn set_broadcast(&mut self, index: usize, value: &[u8]) {
        debug_assert!(index < self.count);
        debug_assert!(value.len() <= self.operand_size);
        let base = index * self.operand_size;
        let mut i = 0;
        while i < self.operand_size {
            let current = base + i;
            self.memory[current..current + value.len()].copy_from_slice(value);

            i += value.len();
        }
    }

    /// Directly copies a part with an offset of `offset` and a size of `size` bytes of an
    /// operand to another.
    #[inline]
    pub fn copy(
        &mut self,
        from_index: usize,
        to_index: usize,
        from_offset: usize,
        to_offset: usize,
        size: usize,
    ) {
        debug_assert!(from_index != to_index);
        debug_assert!(from_offset + size <= self.operand_size);
        debug_assert!(to_offset + size <= self.operand_size);
        let from_address = from_index * self.operand_size + from_offset;
        let to_address = to_index * self.operand_size + to_offset;
        self.memory
            .copy_within(from_address..from_address + size, to_address);
    }

    /// Returns the first `size` bytes from operand `index` as byte slice.
    #[inline]
    pub fn get(&self, index: usize, size: usize) -> &[u8] {
        self.get_offset(index, 0, size)
    }

    /// Returns a byte slice of operand `index`, starting at `offset`, with a size of `size` bytes.
    #[inline]
    pub fn get_offset(&self, index: usize, offset: usize, size: usize) -> &[u8] {
        debug_assert!(index < self.count);
        debug_assert!(offset + size <= self.operand_size);
        let base = index * self.operand_size + offset;
        &self.memory[base..base + size]
    }

    /// Returns the first `f32` element of operand `index`.
    #[inline]
    pub fn get_f32(&self, index: usize) -> f32 {
        self.get_offset_f32(index, 0)
    }

    /// Returns the `f32` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_f32(&self, index: usize, offset: usize) -> f32 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        f32::from_bits(u32::from_ne_bytes(
            self.memory[base..base + 4].try_into().unwrap(),
        ))
    }

    /// Returns the first `f64` element of operand `index`.
    #[inline]
    pub fn get_f64(&self, index: usize) -> f64 {
        self.get_offset_f64(index, 0)
    }

    /// Returns the `f64` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_f64(&self, index: usize, offset: usize) -> f64 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        f64::from_bits(u64::from_ne_bytes(
            self.memory[base..base + 8].try_into().unwrap(),
        ))
    }

    /// Returns the first `i8` element of operand `index`.
    #[inline]
    pub fn get_i8(&self, index: usize) -> i8 {
        self.get_offset_i8(index, 0)
    }

    /// Returns the `i8` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_i8(&self, index: usize, offset: usize) -> i8 {
        debug_assert!(index < self.count);
        debug_assert!(offset < self.operand_size);
        self.memory[index * self.operand_size + offset] as i8
    }

    /// Returns the first `u8` element of operand `index`.
    #[inline]
    pub fn get_u8(&self, index: usize) -> u8 {
        self.get_offset_u8(index, 0)
    }

    /// Returns the `u8` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_u8(&self, index: usize, offset: usize) -> u8 {
        debug_assert!(index < self.count);
        debug_assert!(offset < self.operand_size);
        self.memory[index * self.operand_size + offset]
    }

    /// Returns the first `i16` element of operand `index`.
    #[inline]
    pub fn get_i16(&self, index: usize) -> i16 {
        self.get_offset_i16(index, 0)
    }

    /// Returns the `i16` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_i16(&self, index: usize, offset: usize) -> i16 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 2 <= self.operand_size);
        let base = index * self.operand_size + offset;
        i16::from_ne_bytes(self.memory[base..base + 2].try_into().unwrap())
    }

    /// Returns the first `u16` element of operand `index`.
    #[inline]
    pub fn get_u16(&self, index: usize) -> u16 {
        self.get_offset_u16(index, 0)
    }

    /// Returns the `u16` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_u16(&self, index: usize, offset: usize) -> u16 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 2 <= self.operand_size);
        let base = index * self.operand_size + offset;
        u16::from_ne_bytes(self.memory[base..base + 2].try_into().unwrap())
    }

    /// Returns the first `i32` element of operand `index`.
    #[inline]
    pub fn get_i32(&self, index: usize) -> i32 {
        self.get_offset_i32(index, 0)
    }

    /// Returns the `i32` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_i32(&self, index: usize, offset: usize) -> i32 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        i32::from_ne_bytes(self.memory[base..base + 4].try_into().unwrap())
    }

    /// Returns the first `u32` element of operand `index`.
    #[inline]
    pub fn get_u32(&self, index: usize) -> u32 {
        self.get_offset_u32(index, 0)
    }

    /// Returns the `u32` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_u32(&self, index: usize, offset: usize) -> u32 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        u32::from_ne_bytes(self.memory[base..base + 4].try_into().unwrap())
    }

    /// Returns the first `i64` element of operand `index`.
    #[inline]
    pub fn get_i64(&self, index: usize) -> i64 {
        self.get_offset_i64(index, 0)
    }

    /// Returns the `i64` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_i64(&self, index: usize, offset: usize) -> i64 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        i64::from_ne_bytes(self.memory[base..base + 8].try_into().unwrap())
    }

    /// Returns the first `u64` element of operand `index`.
    #[inline]
    pub fn get_u64(&self, index: usize) -> u64 {
        self.get_offset_u64(index, 0)
    }

    /// Returns the `u64` element of operand `index` at byte offset `offset`.
    #[inline]
    pub fn get_offset_u64(&self, index: usize, offset: usize) -> u64 {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        u64::from_ne_bytes(self.memory[base..base + 8].try_into().unwrap())
    }

    /// Sets the operand to the value of the `value` byte slice. If the slice is shorter than the
    /// operand, the remaining space remains unchanged.
    #[inline]
    pub fn set(&mut self, index: usize, value: &[u8]) {
        self.set_offset(index, 0, value);
    }

    /// Sets the operand at byte offset `offset` to the value of the `value` byte slice. The operand
    /// memory before and behind the slice remains unchanged.
    #[inline]
    pub fn set_offset(&mut self, index: usize, offset: usize, value: &[u8]) {
        debug_assert!(index < self.count);
        debug_assert!(offset + value.len() <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + value.len()].copy_from_slice(value);
    }

    /// Sets the first `f32` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_f32(&mut self, index: usize, value: f32) {
        self.set_offset_f32(index, 0, value);
    }

    /// Sets the `f32` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_f32(&mut self, index: usize, offset: usize, value: f32) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 4].copy_from_slice(&value.to_bits().to_ne_bytes()[..]);
    }

    /// Sets the first `f64` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_f64(&mut self, index: usize, value: f64) {
        self.set_offset_f64(index, 0, value);
    }

    /// Sets the `f64` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_f64(&mut self, index: usize, offset: usize, value: f64) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 8].copy_from_slice(&value.to_bits().to_ne_bytes()[..]);
    }

    /// Sets the first `i8` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_i8(&mut self, index: usize, value: i8) {
        self.set_offset_i8(index, 0, value);
    }

    /// Sets the `i8` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_i8(&mut self, index: usize, offset: usize, value: i8) {
        debug_assert!(index < self.count);
        debug_assert!(offset < self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base] = value as u8;
    }

    /// Sets the first `u8` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_u8(&mut self, index: usize, value: u8) {
        self.set_offset_u8(index, 0, value);
    }

    /// Sets the `u8` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_u8(&mut self, index: usize, offset: usize, value: u8) {
        debug_assert!(index < self.count);
        debug_assert!(offset <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base] = value;
    }

    /// Sets the first `i16` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_i16(&mut self, index: usize, value: i16) {
        self.set_offset_i16(index, 0, value);
    }

    /// Sets the `i16` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_i16(&mut self, index: usize, offset: usize, value: i16) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 2 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 2].copy_from_slice(&value.to_ne_bytes()[..]);
    }

    /// Sets the first `u16` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_u16(&mut self, index: usize, value: u16) {
        self.set_offset_u16(index, 0, value);
    }

    /// Sets the `u16` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_u16(&mut self, index: usize, offset: usize, value: u16) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 2 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 2].copy_from_slice(&value.to_ne_bytes()[..]);
    }

    /// Sets the first `i32` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_i32(&mut self, index: usize, value: i32) {
        self.set_offset_i32(index, 0, value);
    }

    /// Sets the `i32` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_i32(&mut self, index: usize, offset: usize, value: i32) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 4].copy_from_slice(&value.to_ne_bytes()[..]);
    }

    /// Sets the first `u32` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_u32(&mut self, index: usize, value: u32) {
        self.set_offset_u32(index, 0, value);
    }

    /// Sets the `u32` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_u32(&mut self, index: usize, offset: usize, value: u32) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 4 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 4].copy_from_slice(&value.to_ne_bytes()[..]);
    }

    /// Sets the first `i64` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_i64(&mut self, index: usize, value: i64) {
        self.set_offset_i64(index, 0, value);
    }

    /// Sets the `i64` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_i64(&mut self, index: usize, offset: usize, value: i64) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 8].copy_from_slice(&value.to_ne_bytes()[..]);
    }

    /// Sets the first `u64` element of operand `index`. The rest of the operand remains unchanged.
    #[inline]
    pub fn set_u64(&mut self, index: usize, value: u64) {
        self.set_offset_u64(index, 0, value);
    }

    /// Sets the `u64` element of operand `index` at byte offset `offset`. The rest of the operand
    /// remains unchanged.
    #[inline]
    pub fn set_offset_u64(&mut self, index: usize, offset: usize, value: u64) {
        debug_assert!(index < self.count);
        debug_assert!(offset + 8 <= self.operand_size);
        let base = index * self.operand_size + offset;
        self.memory[base..base + 8].copy_from_slice(&value.to_ne_bytes()[..]);
    }
}
