// Filename: elf.rs
// Version:	 0.4
// Date:	 09-07-2021 (DD-MM-YYYY)
// Library:  gpcas_isa
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Contains a general ELF header as well as means for ISA identification.

use crate::isa::IsaIdentifier;

use std::fmt::Formatter;
use std::str::FromStr;

/// ELF identification bytes.
const MAGIC: [u8; 4] = [0x7F, 0x45, 0x4C, 0x46];

/// The first part of an ELF file header.
///
/// This part is the same for all ISAs that use ELF files. It contains the necessary information to
/// select the correct header representation. Different ISAs can have different ELF file header
/// structures beyond this base.
#[derive(Debug, Default)]
#[repr(C)]
pub struct ELFFileHeaderFront {
    /// ELF identification. Should have the value "\x7fELF" when read as string.
    pub magic: [u8; 4],
    /// The ELF file class.
    pub class_id: u8,
    /// The ELF data decoding ID. Should be 1 als the simulator currently only supports little
    /// endian.
    pub data_encoding_id: u8,
    /// The ELF version. Should always be 1 for now.
    pub elf_version: u8,
    /// The used application binary interface.
    pub abi_id: u8,
    /// The version of the ABI.
    pub abi_version: u8,
    /// Unused bytes to align the following fields.
    _padding: [u8; 7],
    /// Type of the ELF file. Should be 2, meaning executable file.
    pub file_type: u16,
    /// The used ISA for the file. This determines which emulator is used and how the ELF file
    /// header might be extended.
    pub machine: u16,
    /// Version of the file.
    pub file_version: u32,
}

/// Error if the lookup of an ISA failed.
#[derive(Debug)]
pub struct UnknownIsaError {
    /// The preformatted error message.
    msg: String,
}

/// Returns the ISA of the program file if it can be identified by the simulator.
pub fn check_isa(executable_data: &[u8]) -> Result<IsaIdentifier, UnknownIsaError> {
    debug_assert!(std::mem::size_of::<ELFFileHeaderFront>() >= 20);
    if executable_data.len() >= std::mem::size_of::<ELFFileHeaderFront>()
        && executable_data[..4] == MAGIC[..]
    {
        let machine_id = unsafe { (executable_data.as_ptr().add(18).cast::<u16>()).read() };

        match machine_id {
            3 => Ok(IsaIdentifier::X86),
            40 => Ok(IsaIdentifier::Arm),
            62 => Ok(IsaIdentifier::X86_64),
            243 => Ok(IsaIdentifier::RiscV),
            0x6233 => Ok(IsaIdentifier::ForwardCom),
            code => Err(UnknownIsaError::from_elf_code(code)), // unimplemented ISA
        }
    } else {
        // No ELF file, default to Arschitek_zero
        Ok(IsaIdentifier::Atz)
    }
}

impl FromStr for IsaIdentifier {
    type Err = UnknownIsaError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.to_ascii_lowercase().as_str() {
            "arm" => Ok(IsaIdentifier::Arm),
            "atz" => Ok(IsaIdentifier::Atz),
            "forwardcom" => Ok(IsaIdentifier::ForwardCom),
            "risc_v" => Ok(IsaIdentifier::RiscV),
            "x86" => Ok(IsaIdentifier::X86),
            "x86_64" => Ok(IsaIdentifier::X86_64),
            e_str => Err(UnknownIsaError::from_str(e_str)),
        }
    }
}

impl UnknownIsaError {
    pub fn from_str(e_str: &str) -> Self {
        UnknownIsaError {
            msg: format!("Unrecognized name \"{}\"", e_str),
        }
    }

    pub fn from_elf_code(code: u16) -> Self {
        UnknownIsaError {
            msg: format!("Unrecognized ELF machine code: {0:#X} or {0}", code),
        }
    }
}

impl std::fmt::Display for UnknownIsaError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.msg)
    }
}

impl std::error::Error for UnknownIsaError {}
