// Filename: isa.rs
// Version:	 0.4
// Date:	 09-07-2021 (DD-MM-YYYY)
// Library:  gpcas_isa
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Data structures to specify instruction set architecture properties.

use crate::emulator::Emulator;

/// How an ISA handles memory access in its instructions.
pub enum MemoryAccessType {
    /// Instructions are either loads, stores or work with registers only.
    LoadStore,
    /// Instructions can load from memory as their last instruction operand.
    RegisterMemory,
}

/// The properties of an instruction set architecture.
pub struct Isa {
    /// Emulates the ISA and sends instruction information to the simulator.
    pub emulator: Box<dyn Emulator>,
    /// How the ISA handles memory accesses.
    pub memory_access_type: MemoryAccessType,
    /// How many registers there are per bank.
    ///
    /// General purpose, vector and special registers each have their own bank.
    pub register_bank_size: usize,
}

/// Identifiers of all ISAs recognized by the simulator.
#[derive(Clone, Copy, Debug)]
pub enum IsaIdentifier {
    /// ARMv8+ (AArch64).
    Arm,
    /// Arschitek_zero, see [`https://git.h3n.eu/Kulasko/arschitek_zero`].
    Atz,
    /// ForwardCom, see [`https://www.forwardcom.info`].
    ForwardCom,
    /// RISC-V, see [`https://riscv.org`].
    RiscV,
    /// 32 bit x86.
    X86,
    /// x86 with 64 bit extension, also called AMD64 or EM64T.
    X86_64,
}
