// Filename: v4.rs
// Version:	 0.2
// Date:	 25-01-2022 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

use crate::{config, CpuModel as NewModel, InstructionLatencies};

use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct CpuModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// The ALU latency of each instruction class.
    pub execution_latencies: InstructionLatencies,
    /// Configuration for the fetch stage of the front end.
    pub fetch: Fetch,
    /// Configuration for the memory controller.
    pub memory_controller: config::MemoryController,
    /// Caches of the memory hierarchy.
    pub caches: Vec<config::Cache>,
    /// Configuration for the decode stage of the front end.
    pub decoder: config::Decoder,
    /// Configuration for the reorder buffer.
    pub reorder_buffer: config::ReorderBuffer,
    /// How many cycles the dispatch unit needs to dispatch an instruction.
    pub dispatch_cycle_count: usize,
    /// Configuration of the register file.
    pub register_file: RegisterFile,
    /// Configuration of schedulers.
    pub schedulers: Vec<config::Scheduler>,
    /// Configuration for each pipeline from just after the front end to the last stage.
    pub pipelines: Vec<config::Pipeline>,
    /// Configuration of the load/store unit.
    pub load_store_unit: LoadStoreUnit,
}

/// Properties of the fetch stage.
///
/// It is responsible for fetching program data from memory.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct Fetch {
    /// The connection to the instruction memory provider.
    pub data_provider: config::MemoryConnection,
    /// Primary branch predictor. Predicts every branch with no additional latency.
    pub primary_predictor: Option<BranchPredictor>,
    /// Secondary branch predictor. Predicts every branch with settable latency. Its result has
    /// priority over the primary predictor.
    pub secondary_predictor: Option<(BranchPredictor, usize)>,
    /// If branch resolution in the front end is enabled.
    ///
    /// After loading the data from cache, the front end will attempt calculate the branch address
    /// if it isn't dependent on registers or memory. Unconditional jumps will cause a redirection,
    /// if not correctly predicted already.
    pub branch_resolution: bool,
}

/// Properties of a branch predictor.
#[derive(Clone, Deserialize, Serialize)]
pub enum BranchPredictor {
    /// Multi-part branch predictor.
    BranchTargetBuffer {
        /// The capacity of saved branches.
        entries: usize,
        /// If bi-modal tables are used to predict conditional branches.
        bimodal_prediction: bool,
        /// Limit of call depth. Unlimited if not set.
        return_address_stack_size: Option<usize>,
    },
    /// TAGE branch predictor. Doesn't provide addresses. Only predicts conditional branches.
    TaggedGeometric {
        /// Prediction tables of geometrically increasing history length.
        tables: Vec<config::TaggedGeometricTable>,
        /// The history length of the first table.
        table_history_length_start: usize,
        /// The factor the history length grows for each table.
        table_history_length_factor: f64,
    },
}

/// The properties of the register file.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct RegisterFile {
    /// If the register file supports renaming.
    pub renaming: bool,
    /// The number of physical general purpose registers.
    ///
    /// If not set, defaults to the number of architectural general purpose registers.
    pub general_purpose_register_count: Option<usize>,
    /// The number of physical vector registers.
    ///
    /// If not set, defaults to the number of architectural vector registers.
    pub vector_register_count: Option<usize>,
}

/// Properties of the load-store-unit.
///
/// It is responsible for connecting the core to the memory hierarchy.
/// It manages in-flight requests.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct LoadStoreUnit {
    /// The amount of load requests that can be in-flight.
    pub load_count: usize,
    /// The amount of store requests that can be in-flight.
    pub store_count: usize,
    /// Connection to the provider of data memory.
    pub data_provider: config::MemoryConnection,
}

impl From<CpuModel> for NewModel {
    fn from(old: CpuModel) -> Self {
        let CpuModel {
            max_vector_size,
            execution_latencies,
            fetch,
            memory_controller,
            caches,
            decoder,
            reorder_buffer,
            dispatch_cycle_count,
            register_file,
            schedulers,
            pipelines,
            load_store_unit,
        } = old;

        Self {
            max_vector_size,
            execution_latencies,
            fetch: fetch.into(),
            memory_controller,
            caches,
            decoder,
            reorder_buffer,
            dispatch_cycle_count,
            register_file: register_file.into(),
            schedulers,
            pipelines,
            load_store_unit: load_store_unit.into(),
        }
    }
}

impl From<Fetch> for config::Fetch {
    fn from(old: Fetch) -> Self {
        let Fetch {
            data_provider,
            primary_predictor,
            secondary_predictor,
            branch_resolution,
        } = old;

        Self {
            data_provider,
            primary_predictor: primary_predictor.map(|predictor| predictor.into()),
            secondary_predictor: secondary_predictor
                .map(|(predictor, latency)| (predictor.into(), latency)),
            branch_resolution,
        }
    }
}

impl From<BranchPredictor> for config::BranchPredictor {
    fn from(old: BranchPredictor) -> Self {
        match old {
            BranchPredictor::BranchTargetBuffer {
                entries,
                bimodal_prediction,
                return_address_stack_size,
            } => Self::BranchTargetBuffer {
                entries,
                bimodal_prediction,
                return_address_stack_size,
            },
            BranchPredictor::TaggedGeometric {
                tables,
                table_history_length_start,
                table_history_length_factor,
            } => {
                let table_history_length_high = (table_history_length_start as f64
                    * table_history_length_factor.powi(tables.len() as i32)
                    + 0.5) as usize;
                Self::TaggedGeometric {
                    tables,
                    table_history_length_low: table_history_length_start,
                    table_history_length_high,
                }
            }
        }
    }
}

impl From<RegisterFile> for config::RegisterFile {
    fn from(old: RegisterFile) -> Self {
        let RegisterFile {
            general_purpose_register_count,
            vector_register_count,
            ..
        } = old;

        Self {
            general_purpose_register_count,
            vector_register_count,
        }
    }
}

impl From<LoadStoreUnit> for config::LoadStoreUnit {
    fn from(old: LoadStoreUnit) -> Self {
        let LoadStoreUnit {
            load_count,
            store_count,
            data_provider,
        } = old;

        Self {
            load_count,
            store_count,
            data_provider,
            port_count: 1,
        }
    }
}
