// Filename: cache.rs
// Version:	 0.5
// Date:	 01-01-2021 (DD-MM-YYYY)
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Configuration for cache components.

use crate::config::MemoryConnection;

use serde::{Deserialize, Serialize};

/// Properties to describe a cache.
///
/// Its role is to cache memory accesses for a faster average response time.
/// As caches get very big (and power hungry) very fast, there are many way to tweak them to
/// perfectly fit a particular purpose.
#[derive(Clone, Deserialize, Serialize)]
pub struct Cache {
    /// Identification number, used for connecting caches together.
    pub id: usize,
    /// The size in power-of-two bytes.
    pub size: usize,
    /// The size of a cache line power-of-two bytes.
    pub line_size: usize,
    /// How many lines are in each set as power-of-two. A line can be saved in any position within
    /// its set.
    pub associativity: usize,
    /// The latency in clock cycles from receiving a request to sending the response.
    pub access_latency: usize,
    /// How the cache handles its CPU side ports.
    pub cache_type: CacheType,
    /// How many misses can be outstanding before the cache stalls.
    pub max_outstanding_misses: usize,
    /// How many cache lines can be outstanding before the cache stalls.
    pub line_buffer_size: usize,
    /// How much data can be transferred to the CPU side in power-of-two bytes per clock.
    pub bandwidth: u16,
    /// If the cache has a next-line prefetcher.
    pub prefetch: bool,
    /// The connection to the memory provider of the next layer.
    pub data_provider: MemoryConnection,
}

/// How a cache handles its CPU side ports.
#[derive(Clone, Deserialize, Serialize)]
pub enum CacheType {
    /// Only one port can access the cache in any clock cycle. Priority is in
    /// descending order. Prefer this if you don't need multiple cache ports.
    Multiplexed,
    /// Each port can access the cache concurrently.
    MultiPorted,
}
