// Filename: validation.rs
// Version:	 0.1
// Date:	 28-12-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Contains the model validation logic.

use crate::{config, CpuModel};
use std::fmt::Formatter;

use gpcas_isa::Isa;

/// Error signaling a failed validation, carrying individual errors inside.
pub struct FailedValidation(Vec<ValidationError>);

/// An individual error of a model that was found during validation.
pub enum ValidationError {
    /// The vector size is no multiple of 128 bits.
    VectorSizeNotMultiple,
    /// An instruction type as a ALU latency of zero clocks, which isn't allowed.
    InstructionZeroLatency,

    /// The fetch unit receives its data from a cache that doesn't exist. Contains cache ID.
    FetchConnectionIdNotFound(usize),
    /// The primary predictor has an invalid configuration. Contains occurred errors.
    FetchInvalidPrimaryPredictor(Vec<BranchPredictionError>),
    /// The secondary predictor has an invalid configuration. Contains occurred errors.
    FetchInvalidSecondaryPredictor(Vec<BranchPredictionError>),
    /// The secondary predictor applies its prediction immediately, which isn't allowed.
    FetchSecondaryPredictorZeroLatency,

    /// The memory controller must have a latency of at least one clock cycle for synchronisation
    /// reasons.
    MemoryControllerZeroLatency,
    /// The memory controller has excessive bandwidth, most likely because it wasn't put in as
    /// power-of-two.
    MemoryControllerTooHighPortWidth,

    /// At least two caches have the same ID, making correctly connecting them impossible.
    CacheIdDuplication,
    /// Contains IDs to caches that create a loop, which doesn't ever resolve to the memory
    /// controller, thus making the caches useless.
    CacheCircularMemoryPath(Vec<usize>),
    /// Excessive cache size, most probably because it wasn't put in as power-of-two. Contains ID of
    /// the offending cache.
    CacheTooLargeSize(usize),
    /// The line size is larger than the entire cache, which isn't possible. Contains ID of the
    /// offending cache.
    CacheLineSizeLargerThanDataSize(usize),
    /// There would have to be more entries in a set than there are in the entire cache, which isn't
    /// possible. Contains ID of the offending cache.
    CacheTooHighAssociativity(usize),
    /// The cache must have a latency of at least one clock cycle for synchronisation reasons.
    /// Contains ID of the offending cache.
    CacheZeroLatency(usize),
    /// Excessive cache bandwidth, most probably because it wasn't put in as power-of-two. Contains
    /// ID of the offending cache.
    CacheTooHighBandwidth(usize),
    /// Cache connects to another cache that doesn't exist. Contains ID of the offending cache and
    /// non-existing ID.
    CacheProviderIdNotFound(usize, usize),

    /// The decoder can't output any instructions.
    DecoderZeroWidth,
    /// The decoder must have its own pipeline stage and as such, does need to take at least one
    /// clock cycle.
    DecoderZeroLatency,

    /// The reorder buffer can't hold any instructions, thus none can be processed.
    ReorderBufferZeroSize,
    /// The reorder buffer must be able to retire at least one instruction per clock, but can't
    /// retire more instructions than can be in-flight at a time.
    ReorderBufferOutOfBoundsWidth,
    /// The reorder buffer must be able to retire at least one branch per clock, but can't
    /// retire more branches than instructions.
    ReorderBufferOutOfBoundsBranchWidth,

    /// The register file holds less general purpose registers than required by the instruction set
    /// architecture.
    RegisterFileTooLowGeneralPurposeRegisters,
    /// The register file holds less vector registers than required by the instruction set
    /// architecture.
    RegisterFileTooLowVectorRegisters,

    /// At least two schedulers have the same ID, making correctly connecting them impossible.
    SchedulerIdDuplication,
    /// The scheduler can't hold entries and thus also don't dispatch any. Contains ID of the
    /// offending scheduler.
    SchedulerZeroEntries(usize),

    /// At least two pipelines have the same ID, which isn't allowed.
    PipelineIdDuplication,
    /// The pipelines connects to a scheduler that doesn't exist. Contains ID of the offending
    /// pipeline and non-existing scheduler ID.
    PipelineSchedulerIdNotFound(usize, usize),
    /// The pipeline doesn't have any instructions it can execute. Contains ID of the offending
    /// pipeline.
    PipelineNoFunction(usize),
    /// The pipeline can execute neither general purpose nor vector instruction, making it useless.
    /// Contains ID of the offending pipeline.
    PipelineNoType(usize),
    /// There's no pipeline capable of executing integer add instructions on general purpose
    /// registers.
    PipelineMissingGeneralPurposeIntegerAdd,
    /// There's no pipeline capable of executing integer multiply instructions on general purpose
    /// registers.
    PipelineMissingGeneralPurposeIntegerMultiply,
    /// There's no pipeline capable of executing integer divide instructions on general purpose
    /// registers.
    PipelineMissingGeneralPurposeIntegerDivide,
    /// There's no pipeline capable of executing integer shift instructions on general purpose
    /// registers.
    PipelineMissingGeneralPurposeIntegerShift,
    /// There's no pipeline capable of executing branch instructions on general purpose registers.
    PipelineMissingGeneralPurposeBranch,
    /// There's no pipeline capable of executing memory loads on general purpose registers.
    PipelineMissingGeneralPurposeLoad,
    /// There's no pipeline capable of executing memory stores on general purpose registers.
    PipelineMissingGeneralPurposeStore,
    /// There's no pipeline capable of executing floating point add instructions, but it's required
    /// by the instruction set.
    PipelineMissingFloatAdd,
    /// There's no pipeline capable of executing floating point multiply instructions, but it's
    /// required by the instruction set.
    PipelineMissingFloatMultiply,
    /// There's no pipeline capable of executing floating point divide instructions, but it's
    /// required by the instruction set.
    PipelineMissingFloatDivide,
    /// There's no pipeline capable of executing integer add instructions on vector registers, but
    /// it's required by the instruction set.
    PipelineMissingVectorIntegerAdd,
    /// There's no pipeline capable of executing integer multiply instructions on vector registers,
    /// but it's required by the instruction set.
    PipelineMissingVectorIntegerMultiply,
    /// There's no pipeline capable of executing integer divide instructions on vector registers,
    /// but it's required by the instruction set.
    PipelineMissingVectorIntegerDivide,
    /// There's no pipeline capable of executing integer shift instructions on vector registers,
    /// but it's required by the instruction set.
    PipelineMissingVectorIntegerShift,
    /// There's no pipeline capable of executing branch instructions on vector registers, but it's
    /// required by the instruction set.
    PipelineMissingVectorBranch,
    /// There's no pipeline capable of executing memory loads on vector registers, but it's
    /// required by the instruction set.
    PipelineMissingVectorLoad,
    /// There's no pipeline capable of executing memory store on vector registers, but it's
    /// required by the instruction set.
    PipelineMissingVectorStore,

    /// The load-store-unit doesn't have any load queue entries and thus can't execute loads.
    LoadStoreUnitZeroLoadEntries,
    /// The load-store-unit doesn't have any store queue entries and thus can't execute stores.
    LoadStoreUnitZeroStoreEntries,
    /// The load-store-unit connects to a cache that doesn't exist. Contains non-existing cache ID.
    LoadStoreUnitConnectionIdNotFound(usize),
}

/// Represents an error inside the configuration of a branch predictor.
pub enum BranchPredictionError {
    /// The return address doesn't have any entries and thus doesn't function.
    ReturnAddressStackZeroEntries,
    /// The branch target buffer doesn't hold any entries and thus doesn't function.
    BranchTargetBufferZeroEntries,
    /// No table holds any history, which makes the predictor non-functioning.
    TaggedGeometricHistoryStartZero,
    /// The history size must increase from one table to the next for the predictor to function.
    /// The set factor is too small to achieve that.
    TaggedGeometricHistoryFactorTooSmall,
    /// The global history grows to excessive values.
    TaggedGeometricHistoryTooLong,
    /// The prediction table contains no value, making it useless. Contains index of the offending
    /// table.
    TaggedGeometricTableZeroEntries(usize),
    /// The prediction table doesn't have an entry count that is a power of two. This is required
    /// because how entries are indexed. Contains ID of the offending table.
    TaggedGeometricTableEntriesNoPowerOfTwo(usize),
    /// The tag of entries inside the table is too small to be useful. Contains index of the
    /// offending table.
    TaggedGeometricTableTooSmallTag(usize),
    /// The tag of entries inside the table is too big. It's an arbitrary limit to achieve fast
    /// simulation.
    TaggedGeometricTableTooLargeTag(usize),
}

/// The maximum global history length of any table of a tagged geometric branch predictor before
/// it's considered excessive.
const MAX_GLOBAL_HISTORY_LENGTH: usize = 1 << 16;
/// The minimum size of entries in a table of a tagged geometric branch predictor before it's
/// considered too small.
const MIN_TAG_SIZE: usize = 2;
/// The maximum size of entries in a table of a tagged geometric branch predictor before it's
/// considered too large.
const MAX_TAG_SIZE: usize = 32;
/// The maximum of valid bandwidth values. Represents in 1^20 bytes -> 1MiB per clock.
const MAX_BANDWIDTH: u16 = 20;
/// The maximum valid cache size. Represents 1^30 bytes -> 1GiB.
const MAX_CACHE_SIZE: usize = 30;
/// Flag to implement optional validation for vector features before the ISA can provide the flag.
const NEEDS_VECTOR_SUPPORT: bool = true;

/// Validates the correctness of a CPU model configuration and returns a list of errors on failure.
pub fn validate(model: &CpuModel, isa: &Isa) -> Result<(), FailedValidation> {
    let mut errors = Vec::new();

    let mut caches = model.caches.iter().collect::<Vec<_>>();
    caches.sort_by_key(|cache| cache.id);
    caches.dedup_by_key(|cache| cache.id);
    let caches = caches; // sealed mutability

    let mut schedulers = model.schedulers.iter().collect::<Vec<_>>();
    schedulers.sort_by_key(|scheduler| scheduler.id);
    schedulers.dedup_by_key(|scheduler| scheduler.id);
    let schedulers = schedulers; // sealed mutability

    let mut pipelines = model.pipelines.iter().collect::<Vec<_>>();
    pipelines.sort_by_key(|pipeline| pipeline.id);
    pipelines.dedup_by_key(|pipeline| pipeline.id);
    let pipelines = pipelines; // sealed mutability

    // execution_latencies
    if model
        .execution_latencies
        .iter()
        .any(|latency| matches!(latency, Some(0)))
    {
        errors.push(ValidationError::InstructionZeroLatency);
    }

    if NEEDS_VECTOR_SUPPORT & (model.max_vector_size == 0 || model.max_vector_size % 128 != 0) {
        errors.push(ValidationError::VectorSizeNotMultiple);
    }

    if let Err(fetch_errors) = validate_fetch(&model.fetch, &caches) {
        errors.extend(fetch_errors);
    }

    if let Err(cache_errors) = validate_caches(&caches, model.caches.len()) {
        errors.extend(cache_errors);
    }

    if let Err(memory_controller_errors) = validate_memory_controller(&model.memory_controller) {
        errors.extend(memory_controller_errors);
    }

    if let Err(decoder_errors) = validate_decoder(&model.decoder) {
        errors.extend(decoder_errors);
    }

    if let Err(reorder_buffer_errors) = validate_reorder_buffer(&model.reorder_buffer) {
        errors.extend(reorder_buffer_errors);
    }

    if let Err(register_file_errors) = validate_register_file(&model.register_file, isa) {
        errors.extend(register_file_errors);
    }

    if let Err(scheduler_errors) = validate_schedulers(&schedulers, model.schedulers.len()) {
        errors.extend(scheduler_errors);
    }

    if let Err(pipeline_errors) =
        validate_pipelines(&pipelines, &schedulers, isa, model.pipelines.len())
    {
        errors.extend(pipeline_errors);
    }

    if let Err(load_store_unit_errors) = validate_load_store_unit(&model.load_store_unit, &caches) {
        errors.extend(load_store_unit_errors);
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(FailedValidation(errors))
    }
}

/// Validates the fetch components. Returns found errors.
fn validate_fetch(
    fetch: &config::Fetch,
    caches: &[&config::Cache],
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if let Some(predictor) = &fetch.primary_predictor {
        if let Err(predictor_errors) = validate_branch_predictor(predictor) {
            errors.push(ValidationError::FetchInvalidPrimaryPredictor(
                predictor_errors,
            ));
        }
    }

    if let Some((predictor, latency)) = &fetch.secondary_predictor {
        if let config::MemoryConnection::Cache(cache_id) = fetch.data_provider {
            if caches
                .binary_search_by_key(&cache_id, |cache| cache.id)
                .is_err()
            {
                errors.push(ValidationError::FetchConnectionIdNotFound(cache_id));
            }
        }

        if let Err(predictor_errors) = validate_branch_predictor(predictor) {
            errors.push(ValidationError::FetchInvalidSecondaryPredictor(
                predictor_errors,
            ));
        }

        if *latency == 0 {
            errors.push(ValidationError::FetchSecondaryPredictorZeroLatency);
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validates a branch predictor. Returns found errors.
fn validate_branch_predictor(
    predictor: &config::BranchPredictor,
) -> Result<(), Vec<BranchPredictionError>> {
    let mut errors = Vec::new();

    match predictor {
        config::BranchPredictor::BranchTargetBuffer {
            entries,
            return_address_stack_size,
            ..
        } => {
            if *entries == 0 {
                errors.push(BranchPredictionError::BranchTargetBufferZeroEntries);
            }

            if let Some(0) = *return_address_stack_size {
                errors.push(BranchPredictionError::ReturnAddressStackZeroEntries);
            }
        }
        config::BranchPredictor::TaggedGeometric {
            tables,
            table_history_length_start,
            table_history_length_factor,
        } => {
            let length_start = *table_history_length_start;
            let length_factor = *table_history_length_factor;

            for (i, table) in tables.iter().enumerate() {
                if table.tag_size < MIN_TAG_SIZE {
                    errors.push(BranchPredictionError::TaggedGeometricTableTooSmallTag(i));
                } else if table.tag_size > MAX_TAG_SIZE {
                    errors.push(BranchPredictionError::TaggedGeometricTableTooLargeTag(i));
                }

                if table.entries == 0 {
                    errors.push(BranchPredictionError::TaggedGeometricTableZeroEntries(i));
                } else if !table.entries.is_power_of_two() {
                    errors.push(BranchPredictionError::TaggedGeometricTableEntriesNoPowerOfTwo(i));
                }
            }

            if length_start == 0 {
                errors.push(BranchPredictionError::TaggedGeometricHistoryStartZero);
            }

            if ((length_start as f64 * length_factor + 0.5) as usize) < length_start + 1 {
                errors.push(BranchPredictionError::TaggedGeometricHistoryFactorTooSmall);
            }

            if ((length_start as f64 * length_factor.powi(tables.len() as i32) + 0.5) as usize)
                > MAX_GLOBAL_HISTORY_LENGTH
            {
                errors.push(BranchPredictionError::TaggedGeometricHistoryTooLong);
            }
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validates all caches. Returns found errors.
fn validate_caches(
    caches: &[&config::Cache],
    gross_cache_count: usize,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if caches.len() < gross_cache_count {
        errors.push(ValidationError::CacheIdDuplication);
    }

    for cache in caches.iter().copied() {
        if cache.size > MAX_CACHE_SIZE {
            errors.push(ValidationError::CacheTooLargeSize(cache.id));
        }

        if cache.line_size > cache.size {
            errors.push(ValidationError::CacheLineSizeLargerThanDataSize(cache.id));
        } else if cache.associativity > cache.size - cache.line_size {
            errors.push(ValidationError::CacheTooHighAssociativity(cache.id));
        }

        if cache.access_latency == 0 {
            errors.push(ValidationError::CacheZeroLatency(cache.id));
        }

        if cache.bandwidth > MAX_BANDWIDTH {
            errors.push(ValidationError::CacheTooHighBandwidth(cache.id));
        }

        if let config::MemoryConnection::Cache(provider_id) = cache.data_provider {
            if caches
                .binary_search_by_key(&provider_id, |provider| provider.id)
                .is_err()
            {
                errors.push(ValidationError::CacheProviderIdNotFound(
                    cache.id,
                    provider_id,
                ));
            }
        }
    }

    let mut taken_part = vec![false; caches.len()];
    for (i, cache) in caches.iter().copied().enumerate() {
        if !taken_part[i] {
            taken_part[i] = true;
            if let config::MemoryConnection::Cache(provider_id) = cache.data_provider {
                if let Ok(connection_index) =
                    caches.binary_search_by_key(&provider_id, |cache| cache.id)
                {
                    let mut current = caches[connection_index];
                    let mut chain = vec![cache.id];

                    if !std::ptr::eq(current, cache) {
                        taken_part[connection_index] = true;
                        chain.push(current.id);
                        while let (config::MemoryConnection::Cache(provider_id), false) =
                            (current.data_provider, std::ptr::eq(current, cache))
                        {
                            if let Ok(connection_index) =
                                caches.binary_search_by_key(&provider_id, |cache| cache.id)
                            {
                                taken_part[connection_index] = true;
                                current = caches[connection_index];
                                chain.push(current.id);
                            } else {
                                // I didn't want to do this!
                                break;
                            }
                        }

                        if std::ptr::eq(current, cache) {
                            errors.push(ValidationError::CacheCircularMemoryPath(chain));
                        }
                    } else {
                        errors.push(ValidationError::CacheCircularMemoryPath(chain));
                    }
                }
            }
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate the memory controller. Returns found errors.
fn validate_memory_controller(
    memory_controller: &config::MemoryController,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if memory_controller.access_latency == 0 {
        errors.push(ValidationError::MemoryControllerZeroLatency);
    }

    if memory_controller.port_width > MAX_BANDWIDTH {
        errors.push(ValidationError::MemoryControllerTooHighPortWidth);
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate the controller. Returns found errors.
fn validate_decoder(decoder: &config::Decoder) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if decoder.width == 0 {
        errors.push(ValidationError::DecoderZeroWidth);
    }

    if decoder.latency == 0 {
        errors.push(ValidationError::DecoderZeroLatency);
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate the reorder buffer. Returns found errors.
fn validate_reorder_buffer(
    reorder_buffer: &config::ReorderBuffer,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if reorder_buffer.size == 0 {
        errors.push(ValidationError::ReorderBufferZeroSize);
    }

    if reorder_buffer.retire_width == 0 || reorder_buffer.retire_width > reorder_buffer.size {
        errors.push(ValidationError::ReorderBufferOutOfBoundsWidth);
    }

    if reorder_buffer.branch_retire_width == 0
        || reorder_buffer.branch_retire_width > reorder_buffer.retire_width
        || reorder_buffer.branch_retire_width > reorder_buffer.size
    {
        errors.push(ValidationError::ReorderBufferOutOfBoundsBranchWidth);
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate the register file. Returns found errors.
fn validate_register_file(
    register_file: &config::RegisterFile,
    isa: &Isa,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if let Some(registers) = register_file.general_purpose_register_count {
        if registers == 0 || registers < isa.register_bank_size {
            errors.push(ValidationError::RegisterFileTooLowGeneralPurposeRegisters);
        }
    }

    if let Some(registers) = register_file.vector_register_count {
        if NEEDS_VECTOR_SUPPORT & (registers == 0 || registers < isa.register_bank_size) {
            errors.push(ValidationError::RegisterFileTooLowVectorRegisters);
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate all schedulers. Returns found errors.
fn validate_schedulers(
    schedulers: &[&config::Scheduler],
    gross_scheduler_count: usize,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if schedulers.len() < gross_scheduler_count {
        errors.push(ValidationError::SchedulerIdDuplication);
    }

    for scheduler in schedulers.iter().copied() {
        if scheduler.entries == 0 {
            errors.push(ValidationError::SchedulerZeroEntries(scheduler.id));
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate all pipelines. Returns found errors.
fn validate_pipelines(
    pipelines: &[&config::Pipeline],
    schedulers: &[&config::Scheduler],
    _isa: &Isa,
    gross_pipeline_count: usize,
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    let mut general_purpose_integer_add = false;
    let mut general_purpose_integer_multiply = false;
    let mut general_purpose_integer_divide = false;
    let mut general_purpose_integer_shift = false;
    let mut general_purpose_branch = false;
    let mut general_purpose_load = false;
    let mut general_purpose_store = false;
    let mut float_add = false;
    let mut float_multiply = false;
    let mut float_divide = false;
    let mut vector_integer_add = false;
    let mut vector_integer_multiply = false;
    let mut vector_integer_divide = false;
    let mut vector_integer_shift = false;
    let mut vector_branch = false;
    let mut vector_load = false;
    let mut vector_store = false;

    if pipelines.len() < gross_pipeline_count {
        errors.push(ValidationError::PipelineIdDuplication);
    }

    for pipeline in pipelines.iter().copied() {
        if pipeline.general_purpose {
            general_purpose_load |= pipeline.memory_load;
            general_purpose_store |= pipeline.memory_store;

            if let Some(capabilities) = &pipeline.alu {
                general_purpose_integer_add |= capabilities.integer_add;
                general_purpose_integer_multiply |= capabilities.integer_mul;
                general_purpose_integer_divide |= capabilities.integer_divide;
                general_purpose_integer_shift |= capabilities.integer_shift;
                general_purpose_branch |= capabilities.branch;
            }
        }

        if pipeline.vector & NEEDS_VECTOR_SUPPORT {
            vector_load |= pipeline.memory_load;
            vector_store |= pipeline.memory_store;

            if let Some(capabilities) = &pipeline.alu {
                float_add |= capabilities.float_add;
                float_multiply |= capabilities.float_mul;
                float_divide |= capabilities.float_divide;
                vector_integer_add |= capabilities.integer_add;
                vector_integer_multiply |= capabilities.integer_mul;
                vector_integer_divide |= capabilities.integer_divide;
                vector_integer_shift |= capabilities.integer_shift;
                vector_branch |= capabilities.branch;
            }
        }

        if let Some(scheduler_id) = pipeline.scheduler_id {
            if schedulers
                .binary_search_by_key(&scheduler_id, |scheduler| scheduler.id)
                .is_err()
            {
                errors.push(ValidationError::PipelineSchedulerIdNotFound(
                    pipeline.id,
                    scheduler_id,
                ));
            }
        }

        if pipeline.alu.is_none() & !pipeline.memory_load & !pipeline.memory_store {
            errors.push(ValidationError::PipelineNoFunction(pipeline.id));
        }

        if !pipeline.general_purpose & !pipeline.vector {
            errors.push(ValidationError::PipelineNoType(pipeline.id));
        }
    }

    if !general_purpose_integer_add {
        errors.push(ValidationError::PipelineMissingGeneralPurposeIntegerAdd);
    }

    if !general_purpose_integer_multiply {
        errors.push(ValidationError::PipelineMissingGeneralPurposeIntegerMultiply);
    }

    if !general_purpose_integer_divide {
        errors.push(ValidationError::PipelineMissingGeneralPurposeIntegerDivide);
    }

    if !general_purpose_integer_shift {
        errors.push(ValidationError::PipelineMissingGeneralPurposeIntegerShift);
    }

    if !general_purpose_branch {
        errors.push(ValidationError::PipelineMissingGeneralPurposeBranch);
    }

    if !general_purpose_load {
        errors.push(ValidationError::PipelineMissingGeneralPurposeLoad);
    }

    if !general_purpose_store {
        errors.push(ValidationError::PipelineMissingGeneralPurposeStore);
    }

    if NEEDS_VECTOR_SUPPORT {
        if !float_add {
            errors.push(ValidationError::PipelineMissingFloatAdd);
        }

        if !float_multiply {
            errors.push(ValidationError::PipelineMissingFloatMultiply);
        }

        if !float_divide {
            errors.push(ValidationError::PipelineMissingFloatDivide);
        }

        if !vector_integer_add {
            errors.push(ValidationError::PipelineMissingVectorIntegerAdd);
        }

        if !vector_integer_multiply {
            errors.push(ValidationError::PipelineMissingVectorIntegerMultiply);
        }

        if !vector_integer_divide {
            errors.push(ValidationError::PipelineMissingVectorIntegerDivide);
        }

        if !vector_integer_shift {
            errors.push(ValidationError::PipelineMissingVectorIntegerShift);
        }

        if !vector_branch {
            errors.push(ValidationError::PipelineMissingVectorBranch);
        }

        if !vector_load {
            errors.push(ValidationError::PipelineMissingVectorLoad);
        }

        if !vector_store {
            errors.push(ValidationError::PipelineMissingVectorStore);
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

/// Validate the load/store unit. Returns found errors.
fn validate_load_store_unit(
    load_store_unit: &config::LoadStoreUnit,
    caches: &[&config::Cache],
) -> Result<(), Vec<ValidationError>> {
    let mut errors = Vec::new();

    if load_store_unit.load_count == 0 {
        errors.push(ValidationError::LoadStoreUnitZeroLoadEntries);
    }

    if load_store_unit.store_count == 0 {
        errors.push(ValidationError::LoadStoreUnitZeroStoreEntries);
    }

    if let config::MemoryConnection::Cache(cache_id) = load_store_unit.data_provider {
        if caches
            .binary_search_by_key(&cache_id, |cache| cache.id)
            .is_err()
        {
            errors.push(ValidationError::LoadStoreUnitConnectionIdNotFound(cache_id));
        }
    }

    if errors.is_empty() {
        Ok(())
    } else {
        Err(errors)
    }
}

impl std::fmt::Display for FailedValidation {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        writeln!(f, "CPU model validation failed")?;
        writeln!(f, "Caused by:")?;
        for error in &self.0 {
            writeln!(f, "\t{}", error)?;
        }
        Ok(())
    }
}

impl std::fmt::Debug for FailedValidation {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self)
    }
}

impl std::error::Error for FailedValidation {}

impl std::fmt::Display for ValidationError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            ValidationError::VectorSizeNotMultiple => {
                write!(f, "The model vector size isn't a multiple of 128 bits")
            }
            ValidationError::InstructionZeroLatency => {
                write!(
                    f,
                    "At least one instruction has a ALU latency of 0 clock cycles set"
                )
            }
            ValidationError::FetchConnectionIdNotFound(cache_id) => {
                write!(f, "Fetch can't connect, no cache with ID {}", cache_id)
            }
            ValidationError::FetchInvalidPrimaryPredictor(branch_errors) => {
                write!(f, "Invalid primary branch predictor:")?;
                for error in branch_errors {
                    write!(f, "\n\t\t{}", error)?;
                }
                Ok(())
            }
            ValidationError::FetchInvalidSecondaryPredictor(branch_errors) => {
                write!(f, "Invalid secondary (backing) branch predictor:")?;
                for error in branch_errors {
                    write!(f, "\n\t\t{}", error)?;
                }
                Ok(())
            }
            ValidationError::FetchSecondaryPredictorZeroLatency => {
                write!(f, "The secondary (backing) doesn't have a latency set")
            }
            ValidationError::MemoryControllerZeroLatency => {
                write!(
                    f,
                    "The memory controller has a latency of 0 clock cycles set, which isn't allowed"
                )
            }
            ValidationError::MemoryControllerTooHighPortWidth => {
                write!(
                    f,
                    "The memory controller has excessive bandwidth, perhaps it isn't set as a \
                    power of two"
                )
            }
            ValidationError::CacheIdDuplication => {
                write!(f, "ID conflict between at least two cache components")
            }
            ValidationError::CacheCircularMemoryPath(chain) => {
                write!(
                    f,
                    "Circular memory path between cache components: {}",
                    chain[0]
                )?;
                for id in chain.iter().copied().skip(1) {
                    write!(f, " => {}", id)?;
                }
                write!(f, " => {}", chain[0])
            }
            ValidationError::CacheTooLargeSize(id) => {
                write!(
                    f,
                    "The cache with ID {} has excessive bandwidth, perhaps the bandwidth isn't set \
                    as a power of two",
                    *id
                )
            }
            ValidationError::CacheLineSizeLargerThanDataSize(id) => {
                write!(
                    f,
                    "The line size of the cache with ID {} is larger than the cache size",
                    *id
                )
            }
            ValidationError::CacheTooHighAssociativity(id) => {
                write!(
                    f,
                    "The associativity of the cache with ID {} is too high, the set size is higher \
                    than the total size",
                    *id
                )
            }
            ValidationError::CacheZeroLatency(id) => {
                write!(
                    f,
                    "The cache with ID {} has a latency of 0 clock cycles, which isn't allowed",
                    *id
                )
            }
            ValidationError::CacheTooHighBandwidth(id) => {
                write!(
                    f,
                    "The cache with ID {} has excessive bandwidth, perhaps the bandwidth isn't set \
                    as a power of two",
                    *id
                )
            }
            ValidationError::CacheProviderIdNotFound(cache_id, connection_id) => {
                write!(
                    f,
                    "The cache with ID {} connects to a non-existing cache (ID {})",
                    *cache_id, *connection_id
                )
            }
            ValidationError::DecoderZeroWidth => {
                write!(
                    f,
                    "The decoder has a width of 0, making it unable to function"
                )
            }
            ValidationError::DecoderZeroLatency => {
                write!(
                    f,
                    "The decoder has a latency of 0 clock cycles, which isn't allowed"
                )
            }
            ValidationError::ReorderBufferZeroSize => {
                write!(
                    f,
                    "The reorder buffer can't hold any entries and thus doesn't function"
                )
            }
            ValidationError::ReorderBufferOutOfBoundsWidth => {
                write!(
                    f,
                    "The reorder buffer retire width is out of bounds, but be between 1 and the \
                    buffer size"
                )
            }
            ValidationError::ReorderBufferOutOfBoundsBranchWidth => {
                write!(
                    f,
                    "The reorder buffer branch retire width is out of bounds, but be between 1 and \
                    the regular retire width"
                )
            }
            ValidationError::RegisterFileTooLowGeneralPurposeRegisters => {
                write!(
                    f,
                    "The register file has less general purpose registers than required by the \
                    instruction set"
                )
            }
            ValidationError::RegisterFileTooLowVectorRegisters => {
                write!(
                    f,
                    "The register file has less vector registers than required by the instruction \
                    set"
                )
            }
            ValidationError::SchedulerIdDuplication => {
                write!(f, "ID collision between at least two schedulers")
            }
            ValidationError::SchedulerZeroEntries(id) => {
                write!(
                    f,
                    "The scheduler with ID {} doesn't have any entries and thus doesn't function",
                    *id
                )
            }
            ValidationError::PipelineIdDuplication => {
                write!(f, "ID collision between at least two pipelines")
            }
            ValidationError::PipelineSchedulerIdNotFound(pipeline_id, scheduler_id) => {
                write!(
                    f,
                    "The pipeline with ID {} connects to a non-existing scheduler (ID {})",
                    *pipeline_id, *scheduler_id,
                )
            }
            ValidationError::PipelineNoFunction(id) => {
                write!(
                    f,
                    "The pipeline with ID {} has no function it could serve",
                    *id
                )
            }
            ValidationError::PipelineNoType(id) => {
                write!(
                    f,
                    "The pipeline with ID {} uses neither general purpose nor vector registers",
                    *id
                )
            }
            ValidationError::PipelineMissingGeneralPurposeIntegerAdd => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer add instructions \
                    on general purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeIntegerMultiply => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer multiply \
                    instructions on general purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeIntegerDivide => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer divide \
                    instructions on general purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeIntegerShift => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer shift \
                    instructions on general purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeBranch => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing branch instructions on \
                    general purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeLoad => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing memory loads on general \
                    purpose registers"
                )
            }
            ValidationError::PipelineMissingGeneralPurposeStore => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing memory stores on general \
                    purpose registers"
                )
            }
            ValidationError::PipelineMissingFloatAdd => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing floating point add \
                    instructions"
                )
            }
            ValidationError::PipelineMissingFloatMultiply => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing floating point multiply \
                    instructions"
                )
            }
            ValidationError::PipelineMissingFloatDivide => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing floating point divide \
                    instructions"
                )
            }
            ValidationError::PipelineMissingVectorIntegerAdd => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer add instructions \
                    on vector registers"
                )
            }
            ValidationError::PipelineMissingVectorIntegerMultiply => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer multiply \
                    instructions on vector registers"
                )
            }
            ValidationError::PipelineMissingVectorIntegerDivide => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer divide \
                    instructions on vector registers"
                )
            }
            ValidationError::PipelineMissingVectorIntegerShift => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing integer shift \
                    instructions on vector registers"
                )
            }
            ValidationError::PipelineMissingVectorBranch => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing branch instructions on \
                    vector registers"
                )
            }
            ValidationError::PipelineMissingVectorLoad => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing memory loads on vector \
                    registers"
                )
            }
            ValidationError::PipelineMissingVectorStore => {
                write!(
                    f,
                    "The model is missing a pipeline capable of executing memory stores on vector \
                    registers"
                )
            }
            ValidationError::LoadStoreUnitZeroLoadEntries => {
                write!(
                    f,
                    "The load-store-unit has no entries in the load queue, making it unable to \
                    process loads"
                )
            }
            ValidationError::LoadStoreUnitZeroStoreEntries => {
                write!(
                    f,
                    "The load-store-unit has no entries in the store queue, making it unable to \
                    process stores"
                )
            }
            ValidationError::LoadStoreUnitConnectionIdNotFound(cache_id) => {
                write!(
                    f,
                    "The load-store-unit connects to a non-existing cache (ID {})",
                    *cache_id
                )
            }
        }
    }
}

impl std::fmt::Debug for ValidationError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self)
    }
}

impl std::error::Error for ValidationError {}

impl std::fmt::Display for BranchPredictionError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            BranchPredictionError::ReturnAddressStackZeroEntries => {
                write!(
                    f,
                    "The return address stack has zero entries and thus doesn't work"
                )
            }
            BranchPredictionError::BranchTargetBufferZeroEntries => {
                write!(
                    f,
                    "The branch target buffer has zero entries and thus doesn't work"
                )
            }
            BranchPredictionError::TaggedGeometricHistoryStartZero => {
                write!(
                    f,
                    "The geometric history length starts at 0, thus not using any history"
                )
            }
            BranchPredictionError::TaggedGeometricHistoryFactorTooSmall => {
                write!(
                    f,
                    "The geometric growth factor is too small for each table to have a longer \
                    history than the previous one"
                )
            }
            BranchPredictionError::TaggedGeometricHistoryTooLong => {
                write!(
                    f,
                    "The global history length is getting excessively long, consider using less \
                    tables or lowering the growth factor"
                )
            }
            BranchPredictionError::TaggedGeometricTableZeroEntries(id) => {
                write!(f, "Table {} doesn't have any entries", *id)
            }
            BranchPredictionError::TaggedGeometricTableEntriesNoPowerOfTwo(id) => {
                write!(f, "Table {}'s entry count isn't a power of two", *id)
            }
            BranchPredictionError::TaggedGeometricTableTooSmallTag(id) => {
                write!(
                    f,
                    "Table {}'s tag size is too small to be useful, consider using at least {} bits",
                    *id, MIN_TAG_SIZE
                )
            }
            BranchPredictionError::TaggedGeometricTableTooLargeTag(id) => {
                write!(
                    f,
                    "Table {}'s tag size is too large to be useful, consider using at most {} bits",
                    *id, MAX_TAG_SIZE
                )
            }
        }
    }
}

impl std::fmt::Debug for BranchPredictionError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self)
    }
}

impl std::error::Error for BranchPredictionError {}
