// Filename: v3.rs
// Version:	 0.2
// Date:	 31-12-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Data structures for file version 3.

use crate::{config, CpuModel as NewModel, InstructionLatencies};

use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct CPUModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// The ALU latency of each instruction class.
    pub execution_latencies: InstructionLatencies,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: config::MemoryController,
    /// Caches of the memory hierarchy.
    pub caches: Vec<CacheConfig>,
    /// Configuration for the decode stage of the front end.
    pub decode_config: DecodeConfig,
    /// Configuration for the reorder buffer.
    pub reorder_buffer_config: config::ReorderBuffer,
    /// How many cycles the dispatch unit needs to dispatch an instruction.
    pub dispatch_cycle_count: usize,
    /// Configuration of the register file.
    pub reg_file_config: config::RegisterFile,
    /// Configuration of schedulers.
    pub schedulers: Vec<config::Scheduler>,
    /// Configuration for each pipeline from just after the front end to the last stage.
    pub pipelines: Vec<PipelineConfig>,
    /// Configuration of the load-store-unit.
    pub load_store_unit_config: LoadStoreUnitConfig,
}

/// Properties of the fetch stage.
///
/// It is responsible for fetching program data from memory.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct FetchConfig {
    /// The ID of the cache used to fetch instruction data. If no ID is given, connects directly
    /// to the memory controller.
    pub cache_connection: Option<usize>,
    /// Primary branch predictor. Predicts every branch with no additional latency.
    pub primary_predictor: BranchPredictor,
    /// Secondary branch predictor. Predicts every branch with settable latency. Its result has
    /// priority over the primary predictor.
    pub secondary_predictor: Option<(BranchPredictor, usize)>,
    /// If branch resolution in the front end is enabled.
    ///
    /// After loading the data from cache, the front end will attempt calculate the branch address
    /// if it isn't dependent on registers or memory. Unconditional jumps will cause a redirection,
    /// if not correctly predicted already.
    pub branch_resolution: bool,
}

/// Properties of a branch predictor.
#[derive(Clone, Deserialize, Serialize)]
pub enum BranchPredictor {
    /// No branch predictor. Make no prediction.
    None,
    /// Has a static probability to return the correct prediction.
    MagicProbability {
        /// Chance between 0 and 1 for correctly predicting a direct jump or call.
        direct_chance: f64,
        /// Chance between 0 and 1 for correctly predicting a conditional jump.
        conditional_chance: f64,
        /// Chance between 0 and 1 for correctly predicting an indirect jump or call target address.
        address_chance: f64,
        /// Chance between 0 and 1 for correctly predicting a return instruction.
        return_chance: f64,
    },
    /// Multi-part branch predictor.
    BranchTargetBuffer {
        /// The capacity of saved branches.
        entries: usize,
        /// If bi-modal tables are used to predict conditional branches.
        bimodal_prediction: bool,
        /// Limit of call depth. Unlimited if not set.
        return_address_stack_size: Option<usize>,
    },
    /// TAGE branch predictor. Doesn't provide addresses. Only predicts conditional branches.
    ///
    /// Has multiple tables. Each table uses double the history length of the last, starting at two.
    TaggedGeometric(Vec<config::TaggedGeometricTable>),
}

/// Properties to describe a cache.
///
/// Its role is to cache memory accesses for a faster average response time.
/// As caches get very big (and power hungry) very fast, there are many way to tweak them to
/// perfectly fit a particular purpose.
#[derive(Clone, Deserialize, PartialEq, Serialize)]
pub struct CacheConfig {
    /// Identification number, used for connecting caches together.
    pub id: usize,
    /// The size in power-of-two bytes.
    pub size: usize,
    /// The size of a cache line power-of-two bytes.
    pub line_size: usize,
    /// How many lines are in each set as power-of-two. A line can be saved in any position within
    /// its set.
    pub associativity: usize,
    /// The latency in clock cycles from receiving a request to sending the response.
    pub access_latency: usize,
    /// How the cache handles its CPU side ports.
    pub cache_type: CacheType,
    /// How many misses can be outstanding before the cache stalls.
    pub max_outstanding_misses: usize,
    /// How many cache lines can be outstanding before the cache stalls.
    pub line_buffer_size: usize,
    /// How much data can be transferred to the CPU side in power-of-two bytes per clock.
    pub bandwidth: u16,
    /// If the cache has a next-line prefetcher.
    pub prefetch: bool,
    /// Specifies the ID of the cache component that supplies data to this one. If there is none,
    /// the cache get its data from the memory controller.
    pub data_provider_cache: Option<usize>,
}

/// How a cache handles its CPU side ports.
#[derive(Clone, Deserialize, PartialEq, Serialize)]
pub enum CacheType {
    /// Has multiple ports, only one port can access the cache in any clock cycle. Priority is in
    /// descending order.
    Multiplexed(usize),
    /// Has multiple ports, each port can access the cache concurrently.
    MultiPorted(usize),
    /// Has a single port.
    SinglePorted,
}

/// Properties of the decode stage.
///
/// It is responsible for decoding memory into executable instructions.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct DecodeConfig {
    /// How many instructions can be decoded in one clock cycle.
    pub width: usize,
    /// The size of the buffer between fetch and decode stage **in fetch blocks**
    pub pre_decode_buffer_size: usize,
}

/// The properties of a CPU pipeline.
///
/// For more information, see [current `Pipeline`](config::Pipeline).
#[derive(Clone, Deserialize, Serialize)]
pub struct PipelineConfig {
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// How many clock cycles are dedicated to reading register data.
    pub register_read_cycles: usize,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// Which types of ALU instructions the pipeline can support, if any.
    pub alu: Option<config::AluCapabilities>,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

/// Properties of the load-store-unit.
///
/// It is responsible for connecting the core to the memory hierarchy.
/// It manages in-flight requests.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct LoadStoreUnitConfig {
    /// The amount of load requests that can be in-flight.
    pub load_count: usize,
    /// The amount of store requests that can be in-flight.
    pub store_count: usize,
    /// The ID of the cache used to serve memory requests. If no ID is given, connects directly
    /// to the memory controller.
    pub cache_connection: Option<usize>,
}

impl Default for BranchPredictor {
    fn default() -> Self {
        Self::None
    }
}

impl From<CPUModel> for NewModel {
    fn from(old: CPUModel) -> Self {
        let CPUModel {
            max_vector_size,
            execution_latencies,
            fetch_config,
            memory_controller_config,
            caches,
            decode_config,
            reorder_buffer_config,
            dispatch_cycle_count,
            reg_file_config,
            schedulers,
            pipelines,
            load_store_unit_config,
        } = old;

        let caches = caches.into_iter().map(CacheConfig::into).collect();
        let pipelines = pipelines
            .into_iter()
            .enumerate()
            .map(|(i, old)| config::Pipeline {
                id: i,
                scheduler_id: old.scheduler_id,
                register_read_cycles: old.register_read_cycles,
                general_purpose: old.general_purpose,
                vector: old.vector,
                alu: old.alu,
                memory_load: old.memory_load,
                memory_store: old.memory_store,
                renaming: old.renaming,
            })
            .collect();

        Self {
            max_vector_size,
            execution_latencies,
            fetch: fetch_config.into(),
            memory_controller: memory_controller_config,
            caches,
            decoder: decode_config.into(),
            reorder_buffer: reorder_buffer_config,
            dispatch_cycle_count,
            register_file: reg_file_config,
            schedulers,
            pipelines,
            load_store_unit: load_store_unit_config.into(),
        }
    }
}

impl From<FetchConfig> for config::Fetch {
    fn from(old: FetchConfig) -> Self {
        let FetchConfig {
            cache_connection,
            primary_predictor,
            secondary_predictor,
            branch_resolution,
        } = old;

        let secondary_predictor = if let Some((predictor, latency)) = secondary_predictor {
            Option::from(predictor).map(|predictor| (predictor, latency))
        } else {
            None
        };

        Self {
            data_provider: cache_connection.into(),
            primary_predictor: primary_predictor.into(),
            secondary_predictor,
            branch_resolution,
        }
    }
}

impl From<BranchPredictor> for Option<config::BranchPredictor> {
    fn from(old: BranchPredictor) -> Self {
        match old {
            BranchPredictor::None => None,
            BranchPredictor::MagicProbability { .. } => None,
            BranchPredictor::BranchTargetBuffer {
                entries,
                bimodal_prediction,
                return_address_stack_size,
            } => Some(config::BranchPredictor::BranchTargetBuffer {
                entries,
                bimodal_prediction,
                return_address_stack_size,
            }),
            BranchPredictor::TaggedGeometric(tables) => {
                Some(config::BranchPredictor::TaggedGeometric {
                    tables,
                    table_history_length_start: 2,
                    table_history_length_factor: 2.0,
                })
            }
        }
    }
}

impl From<CacheConfig> for config::Cache {
    fn from(old: CacheConfig) -> Self {
        let CacheConfig {
            id,
            size,
            line_size,
            associativity,
            access_latency,
            cache_type,
            max_outstanding_misses,
            line_buffer_size,
            bandwidth,
            prefetch,
            data_provider_cache,
        } = old;

        Self {
            id,
            size,
            line_size,
            associativity,
            access_latency,
            cache_type: cache_type.into(),
            max_outstanding_misses,
            line_buffer_size,
            bandwidth,
            prefetch,
            data_provider: data_provider_cache.into(),
        }
    }
}

impl From<CacheType> for config::CacheType {
    fn from(old: CacheType) -> Self {
        match old {
            CacheType::Multiplexed(_) => Self::Multiplexed,
            CacheType::MultiPorted(_) => Self::MultiPorted,
            CacheType::SinglePorted => Self::MultiPorted,
        }
    }
}

impl From<DecodeConfig> for config::Decoder {
    fn from(old: DecodeConfig) -> Self {
        let DecodeConfig { width, .. } = old;

        Self { width, latency: 1 }
    }
}

impl From<LoadStoreUnitConfig> for config::LoadStoreUnit {
    fn from(old: LoadStoreUnitConfig) -> Self {
        let LoadStoreUnitConfig {
            load_count,
            store_count,
            cache_connection,
        } = old;

        Self {
            load_count,
            store_count,
            data_provider: cache_connection.into(),
        }
    }
}

impl From<Option<usize>> for config::MemoryConnection {
    fn from(old: Option<usize>) -> Self {
        match old {
            Some(id) => Self::Cache(id),
            None => Self::MemoryController,
        }
    }
}
